


/**
 * CarbonGuru Browser Extension - Background Script
 * Handles communication between content scripts and the CarbonGuru backend
 */

// Build timestamp and version derived from manifest to avoid stale hardcoding
const __runtimeApi = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
const __manifest = __runtimeApi.getManifest ? __runtimeApi.getManifest() : { version: '0.0.0', description: '' };
const SCRIPT_VERSION = __manifest.version || '0.0.0';
const __descMatch = (__manifest.description || '').match(/Build\s+(\d{10})/);
const BUILD_TIMESTAMP = __descMatch ? __descMatch[1] : String(Math.floor(Date.now() / 1000));

// ===== CACHE DEBUGGING INFO =====
void 0;
void 0;
void 0;
void 0;
void 0;

// Force reload detection
if (typeof window !== 'undefined' && window.carbonGuruCacheDetection) {
    void 0;
} else if (typeof window !== 'undefined') {
    window.carbonGuruCacheDetection = true;
}
// ===== END CACHE DEBUG =====

        void 0;
void 0;

// Firefox uses native auto-updates only - no manual trigger API available
// Chrome-specific APIs like runtime.requestUpdateCheck() are not supported in Firefox
void 0;
void 0;

class CarbonGuruBackground {
    constructor() {
        // Configuration - default to production, allow URL override for testing
        this.isDevelopment = this.detectDevelopmentMode();
        
        // Initialize WooCommerce basket manager (no local storage)
        this.offsetBasket = new WooCommerceBasketManager();

        // Initialize OAuth manager
        this.oauthManager = new OAuthManager();

        // Initialize validation cache to prevent redundant API calls
        this.validationCache = new Map();
        this.VALIDATION_CACHE_TTL = 30000; // 30 seconds
        
        if (this.isDevelopment) {
            this.backendUrl = 'https://dev.carbonguru.io';
            void 0;
        } else {
            this.backendUrl = 'https://carbonguru.io';
            void 0;
        }
        
        // Get version from manifest instead of hardcoding
        this.extensionVersion = browser.runtime.getManifest().version;
        // Initialize with manifest version, will be updated async
        this.currentVersion = browser.runtime.getManifest().version;
        this.authCredentials = null; // Store user credentials for remote server
        this.fileIntegrityChecks = {
            // Remove placeholder hashes that cause failures
        };
        
        // Track URLs being processed to prevent duplicates
        this.processingUrls = new Set();

        // Add validation cache to prevent redundant validation calls
        this.validationCache = new Map();
        this.VALIDATION_CACHE_TTL = 30000; // 30 seconds

        // Cost cache for offset/removal pricing (5 min TTL)
        this.costCache = new Map();
        this.COST_CACHE_TTL = 5 * 60 * 1000; // 5 minutes

        // User preferences cache for offset/removal products (10 min TTL)
        this.userPreferencesCache = null;
        this.userPreferencesCacheTime = null;
        this.USER_PREFS_CACHE_TTL = 10 * 60 * 1000; // 10 minutes

        // Background script initialization flag for content script synchronization
        this.initialized = false;

        // Track last displayed status message to avoid duplicates during polling
        this.lastDisplayedStatusMessage = null;

        // Session management - track active polling sessions
        this.activeSessions = new Map(); // Map<sessionId, { created: Date, lastPoll: Date, pollIntervalId: number }>
        this.SESSION_MAX_AGE = 20 * 60 * 1000; // 20 minutes in milliseconds

        this.init();
    }

    getEffectiveVersion() {
        return browser.runtime.getManifest().version;
    }

    async hasDataConsent() {
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            const result = await storage.local.get('dataConsentGiven');
            return result.dataConsentGiven === true;
        } catch (e) {
            return false;
        }
    }

    /**
     * Get authentication headers for API requests
     * Returns JWT token from OAuth manager, throws error if not authenticated
     */
    getAuthHeaders() {
        // Check if OAuth manager is available and authenticated
        if (this.oauthManager?.isAuthenticated()) {
            const authHeader = this.oauthManager.getAuthHeader();
            if (authHeader) {
                void 0;
                return {
                    'Content-Type': 'application/json',
                    ...authHeader
                };
            }
        }

        // Authentication not available - provide detailed diagnostic information
        if (!this.oauthManager) {
            console.error('CarbonGuru: ❌ OAuth manager not initialized');
        } else {
            const hasToken = !!this.oauthManager.jwtToken;
            const hasExpiry = !!this.oauthManager.tokenExpiry;

            console.error('CarbonGuru: ❌ Authentication check failed:');
            console.error('  - Token exists:', hasToken);
            console.error('  - Expiry exists:', hasExpiry);

            if (hasExpiry) {
                const now = Date.now();
                const isExpired = now >= this.oauthManager.tokenExpiry;
                const expiryDate = new Date(this.oauthManager.tokenExpiry);

                console.error('  - Token expired:', isExpired);
                console.error('  - Current time:', new Date(now).toISOString());
                console.error('  - Token expiry:', expiryDate.toISOString());
                console.error('  - Time until expiry:', ((this.oauthManager.tokenExpiry - now) / 1000 / 60).toFixed(2), 'minutes');
            }

            console.error('  - isAuthenticated():', this.oauthManager.isAuthenticated());
        }

        // No authentication available - throw error since backend requires JWT
        console.error('CarbonGuru: ❌ No valid JWT token - user must authenticate via WordPress OAuth');
        console.error('CarbonGuru: Please open the extension popup and click "Login with WordPress"');
        throw new Error('Authentication required. Please login via WordPress OAuth in the extension popup.');
    }

    /**
     * Get authentication headers for API requests (async version)
     * Handles service worker suspension by reloading token from storage if needed
     */
    async getAuthHeadersAsync() {
        // Check if OAuth manager is available
        if (!this.oauthManager) {
            console.error('CarbonGuru: ❌ OAuth manager not initialized');
            throw new Error('Authentication required. Please login via WordPress OAuth in the extension popup.');
        }

        // Use async ensureAuthenticated which reloads from storage if in-memory cache is empty
        const isAuthenticated = await this.oauthManager.ensureAuthenticated();

        if (isAuthenticated) {
            const authHeader = this.oauthManager.getAuthHeader();
            if (authHeader) {
                void 0;
                return {
                    'Content-Type': 'application/json',
                    ...authHeader
                };
            }
        }

        // Authentication not available - provide detailed diagnostic information
        const hasToken = !!this.oauthManager.jwtToken;
        const hasExpiry = !!this.oauthManager.tokenExpiry;

        console.error('CarbonGuru: ❌ Authentication check failed (async):');
        console.error('  - Token exists:', hasToken);
        console.error('  - Expiry exists:', hasExpiry);

        if (hasExpiry) {
            const now = Date.now();
            const isExpired = now >= this.oauthManager.tokenExpiry;
            const expiryDate = new Date(this.oauthManager.tokenExpiry);

            console.error('  - Token expired:', isExpired);
            console.error('  - Current time:', new Date(now).toISOString());
            console.error('  - Token expiry:', expiryDate.toISOString());
            console.error('  - Time until expiry:', ((this.oauthManager.tokenExpiry - now) / 1000 / 60).toFixed(2), 'minutes');
        }

        // No authentication available - throw error since backend requires JWT
        console.error('CarbonGuru: ❌ No valid JWT token - user must authenticate via WordPress OAuth');
        console.error('CarbonGuru: Please open the extension popup and click "Login with WordPress"');
        throw new Error('Authentication required. Please login via WordPress OAuth in the extension popup.');
    }

    /**
     * Authenticated fetch with automatic token refresh on 401
     * Wraps fetch() and handles expired tokens by clearing auth and triggering re-login
     * Includes service worker resilience - reloads token from storage if in-memory cache is empty
     */
    async authenticatedFetch(url, options = {}) {
        try {
            // Get auth headers with async storage reload for service worker resilience
            const headers = await this.getAuthHeadersAsync();

            // Merge with provided options
            const fetchOptions = {
                ...options,
                headers: {
                    ...headers,
                    ...(options.headers || {})
                }
            };

            // Make the request
            const response = await fetch(url, fetchOptions);

            // Handle 401 Unauthorized - token expired or invalid
            if (response.status === 401) {
                void 0;
                void 0;

                // Clear the invalid token
                if (this.oauthManager) {
                    await this.oauthManager.logout();
                }

                // Notify user they need to re-authenticate
                const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                runtime.sendMessage({
                    action: 'authExpired',
                    message: 'Your login session has expired. Please log in again.'
                }).catch(err => {
                    void 0;
                });

                // Throw error with helpful message
                throw new Error('Authentication expired. Please log in again via the extension popup.');
            }

            return response;

        } catch (error) {
            // Re-throw authentication errors
            if (error.message && error.message.includes('Authentication')) {
                throw error;
            }

            // Log and re-throw other errors
            console.error('CarbonGuru: Authenticated fetch error:', error);
            throw error;
        }
    }

    detectDevelopmentMode() {
        void 0;

        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const manifest = runtime.getManifest();
            const extensionId = manifest.applications?.gecko?.id || 'unknown';
            const manifestName = manifest.name || 'unknown';

            void 0;
            void 0;

            // Firefox: Production extension IDs: carbonguru@carbonguru.io (current) or carbonguru-prod@carbonguru.io (legacy)
            // Firefox: Development extension ID: carbonguru-dev@carbonguru.io
            // Safari/Chrome: Check manifest name - "CarbonGuru" = prod, "CarbonGuru (Dev)" = dev

            const isProductionById = extensionId === 'carbonguru@carbonguru.io' || extensionId === 'carbonguru-prod@carbonguru.io';
            const isProductionByName = manifestName === 'CarbonGuru';
            const isDevelopmentByName = manifestName.includes('Dev');

            // If we have a Firefox ID, use that; otherwise fall back to manifest name
            if (extensionId !== 'unknown') {
                if (isProductionById) {
                    void 0;
                    return false;
                } else {
                    void 0;
                    return true;
                }
            } else {
                // Safari or Chrome - use manifest name
                if (isDevelopmentByName) {
                    void 0;
                    return true;
                } else if (isProductionByName) {
                    void 0;
                    return false;
                }
            }

            // Safe default: assume production to avoid dev contamination
            void 0;
            return false;
        } catch (error) {
            console.error('CarbonGuru: Error detecting environment:', error);
            // Safe default: assume production to avoid dev contamination
            void 0;
            return false;
        }
    }

    // Legacy method - no longer needed with Extension ID-based detection
    // Environment is automatically determined by extension ID
    switchToDevelopmentMode() {
        void 0;
        void 0;
    }

    async init() {
        void 0;

        // Initialize OAuth manager with environment config
        await this.oauthManager.init({
            backendUrl: this.backendUrl
        });
        void 0;

        // Verify JWT token loaded successfully from storage
        if (this.oauthManager.isAuthenticated()) {
            void 0;
            void 0;
        } else {
            void 0;
        }

        // Set up OAuth callback listener
        this.setupOAuthCallbackListener();

        this.setupMessageListener();

        // Mark initialization as complete - content scripts can now safely check auth status
        this.initialized = true;
        void 0;
        // Firefox-native updates with immediate notification support
        void 0;
        this.startUpdateNotificationListener();
        this.setupBrowserAction();

        // Proactively inject latest content script into all tabs on startup
        try {
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            if (tabsAPI && tabsAPI.query) {
                const tabs = await tabsAPI.query({ url: ['https://*/*'] });
                for (const tab of tabs) {
                    try {
                        // Attempt to ping content script; if it fails, reinject
                        await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tab.id, { action: 'ping' });
                    } catch (_) {
                        // Best-effort reinjection using Manifest V3 APIs where available
                        if (tabsAPI.executeScript) {
                            await tabsAPI.executeScript(tab.id, { file: 'content.js' });
                        }
                    }
                }
            }
        } catch (e) {
            void 0;
        }
        // File integrity check disabled - causing issues
        void 0;

        // Start session cleanup timer (runs every minute)
        this.startSessionCleanupTimer();
    }

    setupMessageListener() {
        // Set up message listener for cross-component communication
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        
        runtime.onMessage.addListener((request, sender, sendResponse) => {
            if (request.action === 'getCachedValidation') {
                const cachedData = this.getCachedValidation(request.url);
                sendResponse(cachedData);
                return true;
            }

            if (request.action === 'validateUrl') {
                this.validateUrl(request.url, sendResponse);
                return true; // Indicates async response
            }

            // Don't handle other actions - let the main message handler process them
            return false;
        });
        
        void 0;
    }

    getCachedValidation(url) {
        const cacheKey = url;
        const cached = this.validationCache.get(cacheKey);
        
        if (cached && Date.now() - cached.timestamp < this.VALIDATION_CACHE_TTL) {
            void 0;
            return cached.response;
        }
        
        return null;
    }

    setCachedValidation(url, response) {
        const cacheKey = url;
        this.validationCache.set(cacheKey, {
            response: response,
            timestamp: Date.now()
        });
        void 0;
    }

    async validateUrl(url, sendResponse) {
        void 0;
        
        // Check cache first
        const cachedResponse = this.getCachedValidation(url);
        if (cachedResponse) {
            sendResponse(cachedResponse);
            return;
        }
        
        try {
            // Get authentication headers from OAuth manager
            const headers = this.getAuthHeaders();

            const response = await fetch(`${this.backendUrl}/validate_url`, {
                method: 'POST',
                headers: headers,
                body: JSON.stringify({ url: url })
            });

            const responseData = await response.json();
            
            // Cache the response
            this.setCachedValidation(url, responseData);
            
            sendResponse(responseData);
        } catch (error) {
            console.error('CarbonGuru: Error validating URL:', error);
            const errorResponse = {
                success: false,
                error: error.message
            };
            sendResponse(errorResponse);
        }
    }


    async validateFileIntegrity() {
        // File integrity check disabled - causing issues
        void 0;
        return true;
    }

    async calculateFileHashes() {
        const hashes = {};
        
        try {
            // Get extension files
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            const files = ['content.js', 'background.js', 'manifest.json'];
            
            for (const filename of files) {
                try {
                    const response = await fetch(runtime.getURL(filename));
                    const content = await response.text();
                    const hash = await this.sha256(content);
                    hashes[filename] = hash;
                } catch (error) {
                    void 0;
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Error calculating file hashes:', error);
        }
        
        return hashes;
    }

    async sha256(message) {
        const msgBuffer = new TextEncoder().encode(message);
        const hashBuffer = await crypto.subtle.digest('SHA-256', msgBuffer);
        const hashArray = Array.from(new Uint8Array(hashBuffer));
        const hashHex = hashArray.map(b => b.toString(16).padStart(2, '0')).join('');
        return `sha256-${hashHex}`;
    }

    notifyFileIntegrityError(filename) {
        console.error(`CarbonGuru Security Alert: File integrity check failed for ${filename}. Please reinstall the extension.`);
    }

    // ==================== Session Management Methods ====================

    /**
     * Start periodic cleanup timer for stale sessions
     * Runs every minute to remove sessions older than 20 minutes
     */
    startSessionCleanupTimer() {
        void 0;

        // Clean up immediately on start
        this.cleanupStaleSessions();

        // Then run every minute
        this.sessionCleanupInterval = setInterval(() => {
            this.cleanupStaleSessions();
        }, 60 * 1000); // 60 seconds
    }

    /**
     * Clean up sessions older than SESSION_MAX_AGE (20 minutes)
     */
    cleanupStaleSessions() {
        const now = Date.now();
        let cleanedCount = 0;

        for (const [sessionId, sessionData] of this.activeSessions.entries()) {
            const age = now - sessionData.created;

            if (age > this.SESSION_MAX_AGE) {
                void 0;

                // Clear the polling interval if it exists
                if (sessionData.pollIntervalId) {
                    clearInterval(sessionData.pollIntervalId);
                }

                // Remove from active sessions
                this.activeSessions.delete(sessionId);
                cleanedCount++;
            }
        }

        if (cleanedCount > 0) {
            void 0;
        }
    }

    /**
     * Cancel all active analysis sessions
     * Stops polling and clears session tracking
     * @returns {Promise<{success: boolean, cancelledCount: number}>}
     */
    async cancelActiveSessions() {
        void 0;

        let cancelledCount = 0;

        for (const [sessionId, sessionData] of this.activeSessions.entries()) {
            void 0;

            // Clear the polling interval
            if (sessionData.pollIntervalId) {
                clearInterval(sessionData.pollIntervalId);
            }

            // Remove from active sessions
            this.activeSessions.delete(sessionId);
            cancelledCount++;
        }

        void 0;

        return {
            success: true,
            cancelledCount: cancelledCount
        };
    }

    /**
     * Get count of active sessions being tracked
     * @returns {Promise<{count: number}>}
     */
    async getActiveSessionCount() {
        const count = this.activeSessions.size;
        void 0;

        return {
            count: count
        };
    }

    /**
     * Add a session to active tracking
     * @param {string} sessionId - Session ID to track
     * @param {number} pollIntervalId - setInterval ID for polling
     */
    trackSession(sessionId, pollIntervalId) {
        void 0;

        this.activeSessions.set(sessionId, {
            created: Date.now(),
            lastPoll: Date.now(),
            pollIntervalId: pollIntervalId
        });
    }

    /**
     * Update last poll time for a session
     * @param {string} sessionId - Session ID to update
     */
    updateSessionPollTime(sessionId) {
        const session = this.activeSessions.get(sessionId);
        if (session) {
            session.lastPoll = Date.now();
        }
    }

    /**
     * Remove a session from tracking (when completed or failed)
     * @param {string} sessionId - Session ID to remove
     */
    removeSession(sessionId) {
        const session = this.activeSessions.get(sessionId);
        if (session) {
            void 0;

            // Clear polling interval
            if (session.pollIntervalId) {
                clearInterval(session.pollIntervalId);
            }

            // Remove from map
            this.activeSessions.delete(sessionId);
        }
    }

    // ==================== End Session Management Methods ====================

    setupBrowserAction() {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        
        // Support both browserAction (manifest v2) and action (manifest v3) APIs
        let actionAPI;
        if (typeof browser !== 'undefined') {
            actionAPI = browser.browserAction || browser.action;
        } else {
            actionAPI = chrome.browserAction || chrome.action;
        }
        
        if (actionAPI && actionAPI.onClicked) {
            actionAPI.onClicked.addListener((tab) => {
                void 0;
                this.showPopup(tab.id);
            });
        } else {
            void 0;
        }
    }

    setupMessageListener() {
        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
        
        runtime.onMessage.addListener((request, sender, sendResponse) => {
            void 0;
            
            switch (request.action) {
                case 'processProduct':
                    this.processProduct(request.data, sendResponse);
                    return true; // Keep message channel open for async response
                
                case 'processProductWithExtraction':
                    this.processProductWithExtraction(request.data, sendResponse, sender);
                    return true; // Keep message channel open for async response
                
                case 'extractWithFullPageContent':
                    this.extractWithFullPageContent(request.data).then(sendResponse);
                    return true; // Keep message channel open for async response
                
                case 'getRecentItems':
                    this.getRecentItems(request.limit || 10).then(sendResponse);
                    return true; // Keep message channel open for async response

                case 'getUserImpact':
                    this.offsetBasket.getUserImpactData().then(sendResponse);
                    return true; // Keep message channel open for async response

                case 'checkForUpdates':
                case 'forceUpdateCheck':
                    this.triggerFirefoxUpdateCheck();
                    sendResponse({ success: true, message: 'Native update check triggered' });
                    return false;
                
                case 'triggerAuthentication':
                    void 0;
                    this.triggerAuthentication(request.credentials).then(sendResponse);
                    return true;

                case 'initiateOAuth':
                    void 0;
                    this.oauthManager.initiateOAuthFlow().then(sendResponse);
                    return true;

                case 'logout':
                    void 0;
                    this.oauthManager.logout().then(sendResponse);
                    return true;

                case 'ping':
                    // Content scripts can check if background is initialized and ready
                    void 0;
                    sendResponse({
                        pong: true,
                        initialized: this.initialized,
                        timestamp: Date.now()
                    });
                    return false; // Sync response

                case 'checkAuthStatus':
                    void 0;
                    // Support cache busting parameter from content scripts
                    this.getAuthStatus(request.bustCache).then(sendResponse);
                    return true; // Async response needed for user info

                case 'getAuthHeader':
                    void 0;
                    const authHeader = this.oauthManager.getAuthHeader();
                    sendResponse({ authHeader: authHeader });
                    return false; // Sync response

                case 'checkGmailStatus':
                    void 0;
                    this.checkGmailConnectionStatus().then(sendResponse);
                    return true; // Async response needed

                case 'processGmailEmail':
                    void 0;
                    this.processGmailEmail(request.messageId, sender.tab.id).then(sendResponse);
                    return true; // Async response needed

                case 'processGmailDateRange':
                    void 0;
                    this.processGmailDateRange(request.options, sender.tab.id).then(sendResponse);
                    return true; // Async response needed

                case 'syncOffsetBasket':
                    // Legacy - now handled automatically via WooCommerce
                    void 0;
                    this.offsetBasket.getBasketSummary().then(basket => {
                        sendResponse({ success: true, message: 'Using WooCommerce cart directly', basket });
                    });
                    return true; // Async response needed

                case 'getVersionInfo':
                    void 0;
                    sendResponse(this.getVersionInfo());
                    return false; // No async response needed

                case 'getCarbonFootprint':
                    void 0;
                this.getCarbonFootprint(request.itemId).then(sendResponse);
                return true;

            case 'detectPageType':
                void 0;
                const pageTypeResult = {
                    isOrderPage: false,
                    isCartPage: false,
                    isProductPage: false,
                    documentType: null,
                    orderNumber: null
                };

                // Check if it's an order page
                const orderPageCheck = this.isOrderPage(request.url, request.pageTitle);
                if (orderPageCheck.is_order_page) {
                    pageTypeResult.isOrderPage = true;
                    pageTypeResult.documentType = orderPageCheck.document_type;

                    // Try to extract order number
                    const orderNum = this.extractOrderNumber(request.url, request.pageTitle);
                    if (orderNum) {
                        pageTypeResult.orderNumber = orderNum;
                    }
                } else {
                    // Check if it's a cart page
                    pageTypeResult.isCartPage = this.isShoppingCartUrl(request.url, request.pageTitle, '');

                    // Check if it's a product page
                    if (!pageTypeResult.isCartPage) {
                        pageTypeResult.isProductPage = this.isProductPage(request.url);
                    }
                }

                sendResponse(pageTypeResult);
                return false; // Sync response

            case 'openPopup':
                void 0;
                this.openExtensionPopup();
                sendResponse({ success: true });
                return true; // Keep message channel open for async response
                
            case 'openPopupFromNotification':
                void 0;
                this.openPopupFromNotification();
                sendResponse({ success: true });
                return false; // No async response needed
                
            case 'mappingCopied':
                void 0;
                // Forward to popup if it's open (best effort)
                this.notifyPopupRefresh(request.data);
                sendResponse({ success: true });
                return false; // No async response needed
                
                case 'validateUrl':
                    void 0;
                    this.validateUrl(request.url, sendResponse);
                    return true; // Keep message channel open for async response

                case 'forceReload':
                    void 0;
                    const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                    runtime.reload();
                    sendResponse({ success: true, message: 'Extension reloading...' });
                    return false; // No async response needed
                
                // WooCommerce Basket Operations
                case 'addToBasket':
                    void 0;
                    this.offsetBasket.addItem(request.item).then(async (result) => {
                        const basket = await this.offsetBasket.getBasketSummary();
                        sendResponse({ ...result, basket });
                    }).catch(error => {
                        console.error('CarbonGuru: Error adding item to basket:', error);
                        sendResponse({ success: false, message: error.message || 'Failed to add item' });
                    });
                    return true; // Keep message channel open for async response

                case 'removeFromBasket':
                    void 0;
                    this.offsetBasket.removeItem(request.cartItemKey).then(async (result) => {
                        const basket = await this.offsetBasket.getBasketSummary();
                        sendResponse({ ...result, basket });
                    }).catch(error => {
                        console.error('CarbonGuru: Error removing item from basket:', error);
                        sendResponse({ success: false, message: error.message || 'Failed to remove item' });
                    });
                    return true; // Keep message channel open for async response

                case 'getBasket':
                    void 0;
                    this.offsetBasket.getBasketSummary().then(basket => {
                        sendResponse({ success: true, basket });
                    });
                    return true; // Async response needed

                case 'openCart':
                    void 0;
                    this.offsetBasket.openCart();
                    sendResponse({ success: true });
                    return false; // No async response needed

                case 'startPolling':
                    void 0;
                    this.offsetBasket.startPolling();
                    sendResponse({ success: true });
                    return false; // No async response needed

                case 'stopPolling':
                    void 0;
                    this.offsetBasket.stopPolling();
                    sendResponse({ success: true });
                    return false; // No async response needed
                
                case 'getOffsetUrl':
                    void 0;
                    this.offsetBasket.getBasket().then(cart => {
                        sendResponse({ success: true, url: cart.cart_url || `${this.backendUrl}/cart` });
                    });
                    return true; // Async response needed

                case 'getOffsetCosts':
                    void 0;
                    this.offsetBasket.getOffsetCosts(request.carbon_kg).then(costs => {
                        sendResponse(costs);
                    }).catch(error => {
                        sendResponse({ success: false, error: error.message });
                    });
                    return true; // Async response needed

                case 'addToBasketWithType':
                    void 0;
                    this.offsetBasket.addToBasketWithType(request.item, request.preferenceType).then(async (result) => {
                        // Fetch updated basket to allow UI refresh (matches addToBasket pattern)
                        const basket = await this.offsetBasket.getBasketSummary();
                        sendResponse({ ...result, basket });
                    }).catch(error => {
                        sendResponse({ success: false, error: error.message });
                    });
                    return true; // Async response needed

                case 'processShoppingCart':
                    void 0;
                    this.processShoppingCart(request.cartData, sendResponse);
                    return true; // Keep message channel open for async response
                
                case 'checkDataConsent':
                    this.hasDataConsent().then(granted => sendResponse({ granted }));
                    return true;

                case 'openConsentPage':
                    const consentUrl = (typeof browser !== 'undefined' ? browser.runtime : chrome.runtime).getURL('consent.html');
                    (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).create({ url: consentUrl });
                    sendResponse({ success: true });
                    return false;

                // Environment Configuration Messages
                case 'getEnvironmentConfig':
                    void 0;
                    const config = {
                        backendUrl: this.backendUrl,
                        isDevelopment: this.isDevelopment,
                        extensionVersion: this.extensionVersion
                    };
                    sendResponse(config);
                    return false; // No async response needed

                case 'getBackendUrl':
                    void 0;
                    sendResponse({ url: this.backendUrl });
                    return false; // No async response needed

                case 'getProductUrl':
                    void 0;
                    // Check if path is already a full URL (starts with http:// or https://)
                    let productUrl;
                    if (request.path.startsWith('http://') || request.path.startsWith('https://')) {
                        // Path is already a full URL, use it as-is
                        productUrl = request.path;
                        void 0;
                    } else {
                        // Path is relative, prepend backend URL
                        const baseUrl = this.backendUrl;
                        const cleanPath = request.path.startsWith('/') ? request.path : `/${request.path}`;
                        productUrl = `${baseUrl}${cleanPath}`;
                        void 0;
                    }
                    sendResponse({ url: productUrl });
                    return false; // No async response needed
                
                case 'getCartSummaryUrl':
                    void 0;
                    // Use WordPress cart summary or order summary permalink based on analysis type
                    let summaryUrl;
                    if (request.analysisType === 'order_page') {
                        summaryUrl = `${this.backendUrl}/order-summary/order-${request.sessionId}/`;
                    } else {
                        summaryUrl = `${this.backendUrl}/cart-summary/cart-${request.sessionId}/`;
                    }
                    sendResponse({ url: summaryUrl });
                    return false; // No async response needed
                
                case 'getOffsetSessionUrl':
                    void 0;
                    const sessionOffsetUrl = `${this.backendUrl}/offset_session/${request.sessionId}`;
                    sendResponse({ url: sessionOffsetUrl });
                    return false; // No async response needed
                
                case 'getReviewUrl':
                    void 0;

                    // Handle both full URLs and relative paths
                    let reviewPath;
                    if (request.detailsUrl.startsWith('http://') || request.detailsUrl.startsWith('https://')) {
                        // Extract pathname from full URL
                        try {
                            const urlObj = new URL(request.detailsUrl);
                            reviewPath = urlObj.pathname.replace('/product/', '/review/');
                        } catch (e) {
                            console.error('CarbonGuru: Invalid URL in getReviewUrl:', request.detailsUrl);
                            reviewPath = request.detailsUrl.replace('/product/', '/review/');
                        }
                    } else {
                        // Already a relative path
                        reviewPath = request.detailsUrl.replace('/product/', '/review/');
                    }

                    const reviewUrl = `${this.backendUrl}${reviewPath}`;
                    void 0;
                    sendResponse({ url: reviewUrl });
                    return false; // No async response needed

                case 'copyExistingMapping':
                    void 0;
                    this.offsetBasket.addItem(request.itemData).then(async (result) => {
                        const basket = await this.offsetBasket.getBasketSummary();
                        sendResponse({ ...result, basket });
                    }).catch(error => {
                        console.error('CarbonGuru: Error adding item to WooCommerce cart:', error);
                        sendResponse({ success: false, error: error.message });
                    });
                    return true; // Keep message channel open for async response

                case 'copyMappingToSession':
                    void 0;
                    (async () => {
                        try {
                            const copyResponse = await fetch(`${this.backendUrl}/copy_existing_mapping`, {
                                method: 'POST',
                                headers: this.getAuthHeaders(),
                                body: JSON.stringify(request.copyData)
                            });
                            if (!copyResponse.ok) {
                                sendResponse({ success: false, error: `HTTP error ${copyResponse.status}` });
                                return;
                            }
                            const data = await copyResponse.json();
                            sendResponse(data);
                        } catch (err) {
                            console.error('CarbonGuru: Error in copyMappingToSession:', err);
                            sendResponse({ success: false, error: err.message || 'Failed to copy mapping' });
                        }
                    })();
                    return true; // Keep message channel open for async response

                // Session Management Operations
                case 'cancelActiveAnalysis':
                    void 0;
                    this.cancelActiveSessions().then(result => {
                        sendResponse(result);
                    }).catch(error => {
                        console.error('CarbonGuru: Error canceling sessions:', error);
                        sendResponse({ success: false, error: error.message });
                    });
                    return true; // Keep message channel open for async response

                case 'getActiveSessionCount':
                    void 0;
                    this.getActiveSessionCount().then(result => {
                        sendResponse(result);
                    }).catch(error => {
                        console.error('CarbonGuru: Error getting session count:', error);
                        sendResponse({ count: 0 });
                    });
                    return true; // Keep message channel open for async response

                default:
                    void 0;
                    sendResponse({ success: false, error: 'Unknown action' });
            }
        });
    }

    async validateUrl(url, sendResponse) {
        try {
            void 0;
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 8000);
            const resp = await fetch(`${this.backendUrl}/validate_url`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ url }),
                signal: controller.signal
            });
            clearTimeout(timeoutId);
            if (!resp.ok) throw new Error(`HTTP ${resp.status}`);
            const data = await resp.json();
            sendResponse({ success: !!data.success, ...data });
        } catch (error) {
            console.error('CarbonGuru: Error validating URL:', error);
            sendResponse({ success: false, error: error.message });
        }
    }

    async extractFullPage(url, sendResponse) {
        try {
            void 0;
            
            const response = await this.sendToBackend('/api/extension/extract-full-page', {
                url: url
            });
            
            if (response.success) {
                sendResponse({
                    success: true,
                    productData: response.productData
                });
            } else {
                sendResponse({
                    success: false,
                    error: response.error
                });
            }
        } catch (error) {
            console.error('CarbonGuru: Error extracting full page:', error);
            sendResponse({
                success: false,
                error: error.message
            });
        }
    }

    async analyzeProduct(url, sendResponse) {
        try {
            void 0;
            
            const response = await this.processProduct({ url, action: 'analyze' });
            
            if (response.success) {
                sendResponse({
                    success: true,
                    results: response.data
                });
            } else {
                sendResponse({
                    success: false,
                    error: response.error
                });
            }
        } catch (error) {
            console.error('CarbonGuru: Error analyzing product:', error);
            sendResponse({
                success: false,
                error: error.message
            });
        }
    }

    showPopup(tabId) {
        const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
        tabsAPI.sendMessage(tabId, { action: 'showPopup' });
    }

    hidePopup(tabId) {
        const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
        tabsAPI.sendMessage(tabId, { action: 'hidePopup' });
    }

    openExtensionPopup() {
        void 0;
        
        // Multiple strategies to open the popup
        try {
            // Strategy 1: Try browserAction.openPopup() directly
            if (typeof browser !== 'undefined' && browser.browserAction && browser.browserAction.openPopup) {
                void 0;
                browser.browserAction.openPopup();
                return;
            } else if (chrome.action && chrome.action.openPopup) {
                void 0;
                chrome.action.openPopup();
                return;
            } else if (chrome.browserAction && chrome.browserAction.openPopup) {
                void 0;
                chrome.browserAction.openPopup();
                return;
            }
            
            // Strategy 2: If direct popup opening fails, try with callback
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            tabsAPI.query({ active: true, currentWindow: true }, (tabs) => {
                if (tabs && tabs.length > 0) {
                    const tab = tabs[0];
                    void 0;
                    
                    try {
                        if (typeof browser !== 'undefined' && browser.browserAction) {
                            browser.browserAction.openPopup({ tabId: tab.id });
                        } else if (chrome.action) {
                            chrome.action.openPopup({ tabId: tab.id });
                        } else if (chrome.browserAction) {
                            chrome.browserAction.openPopup({ tabId: tab.id });
                        }
                    } catch (error) {
                        console.error('CarbonGuru: Popup opening failed:', error);
                        // Strategy 3: Show an alert as last resort
                        alert('CarbonGuru analysis complete! Please click the extension icon to view results.');
                    }
                }
            });
            
        } catch (error) {
            console.error('CarbonGuru: All popup opening strategies failed:', error);
            // Final fallback: show alert
            alert('CarbonGuru analysis complete! Please click the extension icon to view results.');
        }
    }

    openPopupFromNotification() {
        void 0;
        
        // Browser security prevents programmatic popup opening
        // Just log the event for debugging purposes
        void 0;
    }
    

    
    showBackupNotification() {
        void 0;
        
        // Send message to content script to show notification in unified UI
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
            if (tabs && tabs.length > 0) {
                chrome.tabs.sendMessage(tabs[0].id, {
                    action: 'showUnifiedNotification',
                    message: '✅ Analysis Complete! Click the CarbonGuru extension icon above to view your results.',
                    options: {
                        type: 'success',
                        duration: 10000
                    }
                }, (response) => {
                    if (chrome.runtime.lastError) {
                        void 0;
                        // Fallback to browser notification if content script is not available
                        if (chrome.notifications) {
                            chrome.notifications.create({
                                type: 'basic',
                                iconUrl: 'icons/icon48.png',
                                title: 'CarbonGuru Analysis Complete!',
                                message: 'Click the CarbonGuru extension icon to view your carbon footprint results.'
                            });
                        }
                    }
                });
            }
        });
    }

    notifyPopupRefresh(data) {
        void 0;
        
        // This is a best-effort notification to any open popup
        // The popup should handle refreshing its recent items list
        try {
            // Send message to all tabs (popup will receive if open)
            chrome.tabs.query({}, (tabs) => {
                tabs.forEach(tab => {
                    try {
                        chrome.tabs.sendMessage(tab.id, {
                            action: 'refreshAfterMappingCopy',
                            data: data
                        }, () => {
                            // Ignore errors - popup may not be open
                            if (chrome.runtime.lastError) {
                                // Expected - ignore
                            }
                        });
                    } catch (e) {
                        // Expected - ignore
                    }
                });
            });
        } catch (error) {
            void 0;
        }
    }

    async processProduct(productData, sendResponse) {
        try {
            void 0;
            
            const url = productData.url;
            
            // Check if URL is already being processed
            if (this.isUrlBeingProcessed(url)) {
                void 0;
                const errorResponse = {
                    success: false,
                    error: 'This URL is already being processed. Please wait for the current analysis to complete.',
                    isDuplicate: true
                };
                
                if (sendResponse) {
                    sendResponse(errorResponse);
                }
                return errorResponse;
            }
            
            // Mark URL as being processed
            this.markUrlAsProcessing(url);
            
            try {
                // Check if we have authentication for API calls - use OAuth manager
                if (!this.oauthManager.isAuthenticated()) {
                    void 0;
                    throw new Error('Authentication required. Please log in via WordPress OAuth in the extension popup.');
                }
                
                const response = await this.sendToBackend(productData);
                
                if (response.success) {
                    void 0;
                    const successResponse = {
                        success: true,
                        message: 'Product processing completed successfully',
                        data: response.data
                    };
                    
                    if (sendResponse) {
                        sendResponse(successResponse);
                    }
                    return successResponse;
                } else {
                    void 0;
                    const errorResponse = {
                        success: false,
                        error: response.error || 'Product processing failed'
                    };
                    
                    if (sendResponse) {
                        sendResponse(errorResponse);
                    }
                    return errorResponse;
                }
            } finally {
                // Always remove URL from processing set
                this.removeUrlFromProcessing(url);
            }
            
        } catch (error) {
            console.error('CarbonGuru: Error processing product:', error);
            
            // Remove URL from processing set on error
            if (productData && productData.url) {
                this.removeUrlFromProcessing(productData.url);
            }
            
            const errorResponse = {
                success: false,
                error: error.message || 'An error occurred during product processing'
            };
            
            if (sendResponse) {
                sendResponse(errorResponse);
            }
            return errorResponse;
        }
    }

    async processProductWithExtraction(productData, sendResponse, sender = null) {
        try {
            void 0;

            const { url, action = 'analyze', page_title, extracted_at, extraction_method } = productData;
            let full_page_content = productData.full_page_content;

            // Check authentication FIRST before any processing - use OAuth manager
            if (!this.oauthManager.isAuthenticated()) {
                void 0;

                // Send message to content script to show authentication prompt
                try {
                    const tabs = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).query({active: true, currentWindow: true});
                    if (tabs[0]) {
                        await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                            action: 'showAuthenticationRequired',
                            message: 'Please log in via WordPress OAuth to analyze this product',
                            url: url
                        });
                    }
                } catch (error) {
                    void 0;
                }

                sendResponse({
                    success: false,
                    error: 'Authentication required. Please log in to analyze products.',
                    needsAuthentication: true
                });
                return;
            }

            // Check if URL is already being processed
            if (this.isUrlBeingProcessed(url)) {
                void 0;
                sendResponse({
                    success: false,
                    error: 'This URL is already being processed. Please wait for the current analysis to complete.',
                    isDuplicate: true
                });
                return;
            }
            
                    // First, validate URL to check for existing mappings (with caching)
        try {
            void 0;
            
            // Check validation cache first
            const now = Date.now();
            const cached = this.validationCache.get(url);
            
            let validationResponse;
            if (cached && (now - cached.timestamp) < this.VALIDATION_CACHE_TTL) {
                void 0;
                validationResponse = { 
                    ok: true, 
                    json: () => Promise.resolve(cached.data) 
                };
            } else {
                void 0;
                
                // Include page content for cross-URL matching
                const validationPayload = { 
                    url: url,
                    page_content: full_page_content || ''
                };
                
                validationResponse = await fetch(`${this.backendUrl}/validate_url`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(validationPayload)
                });
                
                // Cache the response if successful
                if (validationResponse.ok) {
                    const responseData = await validationResponse.json();
                    this.validationCache.set(url, {
                        data: responseData,
                        timestamp: now
                    });
                    validationResponse = { 
                        ok: true, 
                        json: () => Promise.resolve(responseData) 
                    };
                }
            }

                if (validationResponse.ok) {
                    const validationData = await validationResponse.json();
                    void 0;
                    
                    if (validationData.existing_items && validationData.existing_items.length > 0) {
                        void 0;
                        
                        // Send existing mappings to content script for user choice dialog
                        try {
                            const tabs = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).query({active: true, currentWindow: true});
                            if (tabs[0]) {
                                const response = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                                    action: 'showExistingMappingsDialog',
                                    existingItems: validationData.existing_items,
                                    url: url
                                });
                                
                                if (response && response.choice === 'use_existing') {
                                    void 0;
                                    
                                    // Send the existing item data to content script for display
                                    const selectedItem = validationData.existing_items[0]; // Use first item
                                    
                                    // Ensure proper details_url format
                                    if (selectedItem && !selectedItem.details_url && selectedItem.session_id && selectedItem.carbonguru_id) {
                                        selectedItem.details_url = `/product/${selectedItem.session_id}/${selectedItem.carbonguru_id}`;
                                    }
                                    
                                    try {
                                        // Copy the existing mapping first
                                        const copyResponse = await fetch(`${this.backendUrl}/copy_existing_mapping`, {
                                            method: 'POST',
                                            headers: this.getAuthHeaders(),
                                            body: JSON.stringify({
                                                source_item_id: selectedItem.id,
                                                source_session_id: selectedItem.session_id,
                                                current_session_id: 'extension_session',
                                                url: url
                                            })
                                        });
                                        
                                        if (copyResponse.ok) {
                                            void 0;
                                        } else {
                                            void 0;
                                        }
                                        
                                        // Send analysisComplete message to display existing item
                                        await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                                            action: 'analysisComplete',
                                            success: true,
                                            data: selectedItem,
                                            useExisting: true
                                        });
                                        
                                        void 0;
                                    } catch (displayError) {
                                        console.error('CarbonGuru: Error handling existing mapping:', displayError);
                                    }
                                    
                                    sendResponse({
                                        success: true,
                                        useExisting: true,
                                        existingItems: validationData.existing_items
                                    });
                                    return;
                                } else if (response && response.choice === 'remap') {
                                    void 0;
                                    // Set flag to bypass duplicate check in the API call
                                    productData.bypass_duplicate_check = true;
                                } else {
                                    void 0;
                                    // Set flag to bypass duplicate check in the API call
                                    productData.bypass_duplicate_check = true;
                                }
                            }
                        } catch (contentScriptError) {
                            void 0;
                        }
                    } else {
                        void 0;
                    }
                } else {
                    void 0;
                    try {
                        const errorText = await validationResponse.text();
                        void 0;
                    } catch (e) {
                        void 0;
                    }
                }
            } catch (validationError) {
                void 0;
            }
            
            // Mark URL as being processed
            this.markUrlAsProcessing(url);
            
            try {
                void 0;
                void 0;
                
                // HTML optimization is handled by the backend preprocessor service
                // Extension sends full HTML - no client-side stripping

                // Try to extract basic product info from the page for display, but don't require it
                let basicProductInfo = {};
                try {
                    // Use DOMParser for safe, inert parsing — no script execution or event handlers
                    const parser = new DOMParser();
                    const doc = parser.parseFromString(full_page_content, 'text/html');

                    // Try to find product name - Updated for Amazon/BestBuy structures
                    const productNameSelectors = [
                        '#productTitle',  // Primary Amazon product title
                        'h1[data-testid="product-title"]', 'h1.product-title', 'h1.product-name',
                        '.product-title h1', '.product-name h1', 'h1:first-of-type',
                        '[data-testid="product-name"]', '.pdp-product-name',
                        'span#productTitle',  // Alternative Amazon title selector
                        // BestBuy common selectors
                        'h1.sku-title', 'div.sku-title > h1', 'h1#sku-title', 'h1[class*="sku-title"]'
                    ];

                    for (const selector of productNameSelectors) {
                        const element = doc.querySelector(selector);
                        if (element && element.textContent.trim()) {
                            basicProductInfo.product_name = element.textContent.trim();
                            break;
                        }
                    }

                    // Try to find price - Updated for Amazon/BestBuy structures
                    const priceSelectors = [
                        '.a-price-whole',  // Amazon's primary price selector
                        '.a-price .a-offscreen',  // Amazon screen reader price
                        'span.a-price-symbol + span.a-price-whole',  // Amazon price structure
                        '[data-testid="price"]', '.price', '.product-price', '.current-price',
                        '.price-current', '.price-now', '.sale-price', '.pdp-price',
                        '.a-price-fraction',  // Amazon price fraction
                        // BestBuy common selectors
                        '.priceView-hero-price span', '.pricing-price__regular-price', 'div.priceView-hero-price span', 'div[class*="price"] span'
                    ];

                    for (const selector of priceSelectors) {
                        const element = doc.querySelector(selector);
                        if (element && element.textContent.trim()) {
                            const priceText = element.textContent.trim();
                            if (priceText.match(/\$\d+|\d+\.\d+/)) {
                                basicProductInfo.price = priceText;
                                break;
                            }
                        }
                    }
                } catch (e) {
                    void 0;
                }
                
                // Send page content and any basic info to analyze endpoint
                void 0;
                const requestBody = {
                    url: url,
                    full_page_content: full_page_content,
                    page_title: page_title,
                    extracted_at: extracted_at,
                    extraction_method: extraction_method,
                    // Include any basic product info we found, but don't require it
                    ...basicProductInfo
                };

                // 📦 ORDER PAGE DETECTION: Check FIRST - more specific than cart detection
                // Order pages often contain cart-like content but should be processed differently
                const orderPageResult = this.isOrderPage(url, page_title);
                if (orderPageResult.is_order_page) {
                    requestBody.analysis_type = 'order_page';
                    requestBody.document_type = orderPageResult.document_type;
                    void 0;

                    // Try to extract order number from URL or page title
                    const orderNumber = this.extractOrderNumber(url, page_title);
                    if (orderNumber) {
                        requestBody.order_number = orderNumber;
                        void 0;
                    }
                } else {
                    // 📋 ORDER DETECTION: Check if this is an order page FIRST (before cart detection)
                    const orderPageResult = this.isOrderPage(url, page_title);
                    if (orderPageResult && orderPageResult.is_order_page) {
                        requestBody.analysis_type = 'order_page';
                        void 0;
                    } else {
                        // 🛒 CART DETECTION: Check if this is a shopping cart URL
                        const isCartUrl = this.isShoppingCartUrl(url, page_title, full_page_content);
                        if (isCartUrl) {
                            requestBody.analysis_type = 'shopping_cart';
                            void 0;
                        } else {
                            // 📦 DEFAULT: Treat as product page
                            // Product is the default - carts/orders require positive detection
                            requestBody.analysis_type = 'product';
                            const isProductUrl = this.isProductPage(url);
                            if (isProductUrl) {
                                void 0;
                            } else {
                                void 0;
                            }
                        }
                    }
                }
                
                // Add bypass flag if user chose to remap existing item
                if (productData.bypass_duplicate_check) {
                    requestBody.bypass_duplicate_check = true;
                    void 0;
                }

                // Use authenticatedFetch which handles 401 errors automatically
                const processResponse = await this.authenticatedFetch(`${this.backendUrl}/api/analyze`, {
                    method: 'POST',
                    body: JSON.stringify(requestBody)
                });

                if (!processResponse.ok) {
                    throw new Error(`Processing submission failed: ${processResponse.status}`);
                }
                const processData = await processResponse.json();
                if (!processData.success) {
                    throw new Error(processData.error || 'Processing submission failed');
                }

                // ✅ DUPLICATE CART: If backend recognized this as a duplicate cart, start polling for immediate results
                // The ActivityLog is already created with analysis_complete, so polling will find results immediately
                if (processData.duplicate_recognized && processData.processing === false) {
                    void 0;
                    // Fall through to normal polling - ActivityLog already exists with analysis_complete
                }

                // ✅ WOOCOMMERCE URL FIX: If response includes wc_product_url, send it to content.js immediately
                // This ensures View Details button uses WooCommerce URL instead of internal URL
                if (processData.wc_product_url && processData.carbonguru_id) {
                    void 0;
                    try {
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        const tabsArray = await tabs.query({ active: true, currentWindow: true });
                        if (tabsArray[0]) {
                            await tabs.sendMessage(tabsArray[0].id, {
                                action: 'updateWooCommerceUrl',
                                carbonguru_id: processData.carbonguru_id,
                                wc_product_url: processData.wc_product_url,
                                wc_product_id: processData.wc_product_id
                            });
                            void 0;
                        }
                    } catch (messageError) {
                        void 0;
                    }
                }

                // Check if this is async cart processing
                if (processData.analysis_type === 'shopping_cart_async' && processData.cart_id) {
                    void 0;
                    
                    // Send status update to content script
                    const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                    tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                        if (tabsArray[0]) {
                            tabs.sendMessage(tabsArray[0].id, {
                                action: 'showUnifiedNotification',
                                message: `Processing large cart (${processData.chunk_count} chunks)...`,
                                options: { 
                                    type: 'progress', 
                                    showProgress: true,
                                    estimatedTime: processData.estimated_time_seconds
                                }
                            }).catch((error) => {
                                void 0;
                            });
                        }
                    });
                    
                    // Start polling for cart results
                    this.pollCartResults(processData.cart_id, url, processData.polling_url);
                    
                    // Send immediate response that we're processing async
                    if (sendResponse) {
                        sendResponse({
                            success: true,
                            message: `Large cart processing started (${processData.chunk_count} chunks) - estimated ${processData.estimated_time_seconds}s`,
                            status: 'processing_async',
                            cart_id: processData.cart_id
                        });
                    }
                    return; // Exit early for async processing
                }
                
                // Check if backend found existing items
                if (processData.has_existing_items && processData.existing_items && processData.existing_items.length > 0) {
                    void 0;
                    
                    // Send existing mappings to content script for user choice dialog
                    try {
                        const tabs = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).query({active: true, currentWindow: true});
                        if (tabs[0]) {
                            const response = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                                action: 'showExistingMappingsDialog',
                                existingItems: processData.existing_items,
                                url: url
                            });
                            
                            if (response && response.choice === 'use_existing') {
                                void 0;
                                
                                // Send the existing item data to content script for display
                                const selectedItem = processData.existing_items[0]; // Use first item
                                
                                // Ensure proper details_url format
                                if (selectedItem && !selectedItem.details_url && selectedItem.session_id && selectedItem.carbonguru_id) {
                                    selectedItem.details_url = `/product/${selectedItem.session_id}/${selectedItem.carbonguru_id}`;
                                }
                                
                                try {
                                    // Copy the existing mapping first
                                    const copyResponse = await fetch(`${this.backendUrl}/copy_existing_mapping`, {
                                        method: 'POST',
                                        headers: this.getAuthHeaders(),
                                        body: JSON.stringify({
                                            source_item_id: selectedItem.id,
                                            source_session_id: selectedItem.session_id,
                                            current_session_id: 'extension_session',
                                            url: url
                                        })
                                    });
                                    
                                    if (copyResponse.ok) {
                                        void 0;
                                    } else {
                                        void 0;
                                    }
                                    
                                    // Send analysisComplete message to display existing item
                                    await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                                        action: 'analysisComplete',
                                        success: true,
                                        data: selectedItem,
                                        useExisting: true
                                    });
                                    
                                    void 0;
                                } catch (displayError) {
                                    console.error('CarbonGuru: Error handling existing mapping:', displayError);
                                }
                                
                                sendResponse({
                                    success: true,
                                    useExisting: true,
                                    existingItems: processData.existing_items
                                });
                                return;
                            } else if (response && response.choice === 'remap') {
                                void 0;
                                // Set flag to bypass duplicate check in the API call
                                productData.bypass_duplicate_check = true;
                                // Continue with processing below
                            } else {
                                void 0;
                            }
                        }
                    } catch (contentScriptError) {
                        void 0;
                    }
                }
                
                const process_id = processData.process_id;

                void 0;
                
                // Send status update to content script that processing has started
                const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                    if (tabsArray[0]) {
                        tabs.sendMessage(tabsArray[0].id, {
                            action: 'showUnifiedNotification',
                            message: 'Analyzing Carbon footprint...',
                            options: { type: 'progress', showProgress: true }
                        }).catch((error) => {
                            void 0;
                        });
                    }
                });
                
                // Start polling database for this URL's results
                const senderTabId = sender && sender.tab ? sender.tab.id : null;
                const backendAnalysisType = processData.analysis_type; // Use backend's determination
                // ✅ WOOCOMMERCE URL FIX: Pass wc_product_url to pollDatabaseForResults
                const initialWcData = {
                    wc_product_url: processData.wc_product_url,
                    wc_product_id: processData.wc_product_id,
                    carbonguru_id: processData.carbonguru_id
                };
                this.pollDatabaseForResults(process_id, url, senderTabId, backendAnalysisType, initialWcData);
                
                // Send immediate response that we're processing
                if (sendResponse) {
                    sendResponse({
                        success: true,
                        message: 'Analysis started using captured page content - polling database for results...',
                        status: 'processing'
                    });
                }
            } finally {
                // Always remove URL from processing set
                this.removeUrlFromProcessing(url);
            }
            
        } catch (error) {
            console.error('CarbonGuru: Error processing product with extraction:', error);
            
            // Remove URL from processing set on error
            if (productData && productData.url) {
                this.removeUrlFromProcessing(productData.url);
            }
            
            // IMPORTANT: Even if extension processing failed, the backend might have processed via fallback
            // Check if the backend processed the URL successfully despite the extension error
            try {
                void 0;
                await new Promise(resolve => setTimeout(resolve, 3000)); // Wait 3 seconds for backend processing
                
                const checkResponse = await this.sendToBackend('/api/mapped-items', 'GET', null, { session_id: 'extension_content_session' });
                
                if (checkResponse && checkResponse.success && checkResponse.items) {
                    // Check if any recent item matches this URL
                    const recentItem = checkResponse.items.find(item => 
                        item.original_url && productData.url && 
                        item.original_url.includes(new URL(productData.url).pathname.split('/').pop())
                    );
                    
                    if (recentItem) {
                        void 0;
                        
                        // Send analysisComplete message to content script to update floating tile
                        try {
                            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                            tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                                if (tabsArray[0]) {
                                    tabs.sendMessage(tabsArray[0].id, {
                                        action: 'analysisComplete',
                                        success: true,
                                        data: {
                                            product_carbon_footprint: recentItem.product_carbon_footprint,
                                            product_name: recentItem.product_name,
                                            naics_code: recentItem.naics_code,
                                            naics_title: recentItem.naics_title,
                                            details_url: recentItem.details_url,
                                            type: 'product'
                                        }
                                    });
                                    void 0;
                                }
                            });
                        } catch (messageError) {
                            void 0;
                        }
                        
                        sendResponse({
                            success: true,
                            productData: {
                                carbonguru_id: recentItem.carbonguru_id,
                                product_name: recentItem.product_name,
                                price: recentItem.price,
                                currency: recentItem.currency,
                                product_carbon_footprint: recentItem.product_carbon_footprint,
                                details_url: recentItem.details_url
                            },
                            message: 'Processed successfully via backend fallback'
                        });
                        return;
                    }
                }
            } catch (checkError) {
                void 0;
            }
            
            sendResponse({
                success: false,
                error: error.message || 'An error occurred during product processing'
            });
        }
    }

    async pollAnalysisResults(processId, pollingUrl, originalUrl) {
        void 0;
        
        const maxAttempts = 40; // 40 attempts × 3 seconds = 2 minutes max
        let attempts = 0;
        
        const poll = async () => {
            attempts++;
            
            try {
                const response = await fetch(`${this.getApiBaseUrl()}${pollingUrl}`, {
                    method: 'GET',
                    headers: this.getAuthHeaders()
                });
                
                if (!response.ok) {
                    throw new Error(`Polling failed: ${response.status}`);
                }
                
                const result = await response.json();
                
                if (result.status === 'processing') {
                    // Still processing - send progress update
                    try {
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        const tabsArray = await tabs.query({ active: true, currentWindow: true });
                        if (tabsArray[0]) {
                            await tabs.sendMessage(tabsArray[0].id, {
                                action: 'showUnifiedNotification',
                                message: `Processing analysis... (${attempts}/${maxAttempts})`,
                                options: { 
                                    type: 'progress', 
                                    showProgress: true
                                }
                            });
                        }
                    } catch (messageError) {
                        void 0;
                    }
                    
                    // Continue polling if not exceeded max attempts
                    if (attempts < maxAttempts) {
                        setTimeout(poll, 3000); // Poll every 3 seconds
                    } else {
                        console.error('CarbonGuru: Analysis polling timeout');
                        this.handleAnalysisTimeout(processId, originalUrl);
                    }
                    
                } else if (result.success) {
                    // Analysis completed successfully
                    void 0;
                    
                    // Store processed items in recent items
                    if (result.type === 'cart' && result.items) {
                        for (const item of result.items) {
                            await this.storeRecentItem(item);
                        }
                    } else if (result.type === 'product') {
                        await this.storeRecentItem(result);
                    }
                    
                    // Send analysisComplete message to content script
                    try {
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        const tabsArray = await tabs.query({ active: true, currentWindow: true });
                        if (tabsArray[0]) {
                            await tabs.sendMessage(tabsArray[0].id, {
                                action: 'analysisComplete',
                                success: true,
                                data: result
                            });
                            void 0;
                        }
                    } catch (messageError) {
                        void 0;
                    }
                    
                } else {
                    // Analysis failed
                    console.error('🔍 CarbonGuru: Analysis failed:', result.error);
                    this.handleAnalysisError(processId, originalUrl, result.error);
                }
                
            } catch (error) {
                console.error('🔍 CarbonGuru: Polling error:', error);
                
                if (attempts < maxAttempts) {
                    setTimeout(poll, 3000); // Retry on network errors
                } else {
                    this.handleAnalysisTimeout(processId, originalUrl);
                }
            }
        };
        
        // Start polling after 2 seconds
        setTimeout(poll, 2000);
    }
    
    async handleAnalysisTimeout(processId, originalUrl) {
        console.error('CarbonGuru: Analysis timed out, checking for fallback results');
        
        // Try fallback check similar to existing logic
        try {
            await new Promise(resolve => setTimeout(resolve, 3000));
            
            const checkResponse = await this.sendToBackend('/api/mapped-items', 'GET');
            
            if (checkResponse && checkResponse.success && checkResponse.items) {
                const recentItem = checkResponse.items.find(item => 
                    item.original_url && originalUrl && 
                    item.original_url.includes(new URL(originalUrl).pathname.split('/').pop())
                );
                
                if (recentItem) {
                    // Found result via fallback
                    try {
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        const tabsArray = await tabs.query({ active: true, currentWindow: true });
                        if (tabsArray[0]) {
                            await tabs.sendMessage(tabsArray[0].id, {
                                action: 'analysisComplete',
                                success: true,
                                data: {
                                    product_carbon_footprint: recentItem.product_carbon_footprint,
                                    product_name: recentItem.product_name,
                                    naics_code: recentItem.naics_code,
                                    naics_title: recentItem.naics_title,
                                    details_url: recentItem.details_url,
                                    type: 'product'
                                }
                            });
                            void 0;
                        }
                    } catch (messageError) {
                        void 0;
                    }
                    return;
                }
            }
        } catch (error) {
            void 0;
        }
        
        // No fallback found - show timeout error
        this.handleAnalysisError(processId, originalUrl, 'Analysis timed out');
    }
    
    async handleAnalysisError(processId, originalUrl, error) {
        try {
            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            const tabsArray = await tabs.query({ active: true, currentWindow: true });
            if (tabsArray[0]) {
                await tabs.sendMessage(tabsArray[0].id, {
                    action: 'showUnifiedNotification',
                    message: `Analysis failed: ${error}`,
                    options: { 
                        type: 'error',
                        duration: 4000
                    }
                });
            }
        } catch (messageError) {
            void 0;
        }
    }

    async extractProductData(url) {
        try {
            if (!await this.hasDataConsent()) {
                throw new Error('Data consent required. Please accept the data collection notice.');
            }
            void 0;

            // Check if we have authentication for API calls - use OAuth manager
            if (!this.oauthManager.isAuthenticated()) {
                void 0;
                throw new Error('Authentication required. Please log in via WordPress OAuth in the extension popup.');
            }
            
            // Get authentication headers from OAuth manager
            const headers = this.getAuthHeaders();

            // Increase timeout for problematic sites
            const controller = new AbortController();
            const timeoutId = setTimeout(() => controller.abort(), 30000); // 30 second timeout

            const response = await fetch(`${this.backendUrl}/extension/extract`, {
                method: 'POST',
                headers: headers,
                body: JSON.stringify({ url: url }),
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            
            if (!response.ok) {
                if (response.status === 401) {
                    throw new Error('Authentication required. Please authenticate and try again.');
                } else if (response.status === 408 || response.status === 504) {
                    throw new Error('Request timeout. The website may be slow or blocking requests. Please try again.');
                } else {
                    throw new Error(`HTTP ${response.status}: ${response.statusText}`);
                }
            }
            
            const data = await response.json();
            
            if (data.success) {
                void 0;
                return {
                    success: true,
                    data: data.product_data || data
                };
            } else {
                void 0;
                return {
                    success: false,
                    error: data.error || 'Failed to extract product data'
                };
            }
            
        } catch (error) {
            console.error('CarbonGuru: Error extracting product data:', error);
            
            if (error.name === 'AbortError') {
                return {
                    success: false,
                    error: 'Request timeout. The website may be slow or blocking requests. Please try again.'
                };
            } else if (error.message.includes('timeout')) {
                return {
                    success: false,
                    error: 'Request timeout. The website may be slow or blocking requests. Please try again.'
                };
            } else {
                return {
                    success: false,
                    error: error.message || 'Failed to extract product data'
                };
            }
        }
    }

    async waitForResults_OLD(processId, carbonguruId) {
        void 0;
        
        // Poll for results with extended timeout for major e-commerce sites
        const startTime = Date.now();
        const timeout = 300000; // 5 minutes timeout
        
        while (Date.now() - startTime < timeout) {
            try {
                const response = await this.sendToBackend(`/api/extension/get_job_status/${processId}`, 'GET');
                
                if (response && response.job_status === 'completed') {
                    void 0;
                    // Now, fetch the full results
                    const finalResult = await this.sendToBackend(`/api/extension/get_job_results/${processId}`, 'GET');
                    if (finalResult && finalResult.success) {
                        return finalResult.data; // Return the detailed product data
                    } else {
                        throw new Error('Failed to fetch final results after completion.');
                    }
                } else if (response && response.job_status === 'failed') {
                    throw new Error('Analysis job failed on the server.');
                }
                
                // Wait for a second before polling again
                await new Promise(resolve => setTimeout(resolve, 1000));
                
            } catch (error) {
                console.error(`CarbonGuru: Error while waiting for results for job ${processId}:`, error);
                return { success: false, error: error.message };
            }
        }
        
        throw new Error('Timeout waiting for analysis results.');
    }
    
    async waitForResults(process_id) {
        void 0;
        const startTime = Date.now();
        const timeout = 30000; // 30 seconds
        
        while (Date.now() - startTime < timeout) {
            try {
                // Poll the mapped-items endpoint directly to find our result
                const recentItemsResponse = await this.sendToBackend('/api/mapped-items?limit=50', 'GET');
                
                if (recentItemsResponse.success && recentItemsResponse.items) {
                    const matchedItem = recentItemsResponse.items.find(item => 
                        item.session_id === process_id && 
                        item.product_carbon_footprint !== null && 
                        item.product_carbon_footprint !== undefined
                    );
                    
                    if (matchedItem) {
                        void 0;
                        return { success: true, data: matchedItem };
                    }
                }
                
                await new Promise(resolve => setTimeout(resolve, 2000)); // Wait 2 seconds before polling again
                
            } catch (error) {
                console.error(`CarbonGuru: Error while waiting for results for process ${process_id}:`, error);
                return { success: false, error: error.message };
            }
        }
        
        return { success: false, error: 'Timeout waiting for analysis results.' };
    }

    async extractWithFullPageContent(data) {
        try {
            if (!await this.hasDataConsent()) {
                return { success: false, error: 'Data consent required. Please accept the data collection notice.' };
            }
            void 0;

            // Send full page content to backend for LCA Assistant processing
            const response = await fetch(`${this.backendUrl}/api/extension/extract-full-page`, {
                method: 'POST',
                headers: this.getAuthHeaders(),
                body: JSON.stringify({
                    url: data.url,
                    full_page_content: data.full_page_content,
                    page_title: data.page_title,
                    extracted_at: data.extracted_at,
                    extraction_method: data.extraction_method
                })
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const responseData = await response.json();

            if (responseData.success) {
                void 0;
                
                // Format the product data for further processing
                const productData = {
                    url: data.url,
                    product_name: responseData.product_data.product_name,
                    price: responseData.product_data.price,
                    category: responseData.product_data.category,
                    description: responseData.product_data.description,
                    materials: responseData.product_data.materials,
                    geographic_context: responseData.product_data.geographic_context,
                    extraction_method: 'browser_extension_full_page'
                };
                
                return {
                    success: true,
                    data: productData,  // Return formatted data for processing
                    message: 'Full page extraction completed successfully'
                };
            } else {
                return {
                    success: false,
                    error: responseData.error || 'Full page extraction failed'
                };
            }
        } catch (error) {
            console.error('CarbonGuru: Error in full page extraction:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    async monitorProcessing(processId) {
        try {
            const response = await fetch(`${this.backendUrl}/process_status/${processId}`, {
                method: 'GET',
                headers: this.getAuthHeaders()
            });
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('CarbonGuru: Error monitoring processing:', error);
            return { error: error.message };
        }
    }

    async getProcessedItems() {
        try {
            const response = await fetch(`${this.backendUrl}/api/mapped-items?limit=50`, {
                method: 'GET',
                headers: this.getAuthHeaders()
            });
            
            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            return data;
        } catch (error) {
            console.error('CarbonGuru: Error getting processed items:', error);
            return { error: error.message };
        }
    }

    async getCarbonFootprint(itemId) {
        try {
            void 0;
            const url = `${this.backendUrl}/api/item/${itemId}`;
            void 0;
            
            const response = await fetch(url, {
                method: 'GET',
                headers: this.getAuthHeaders()
            });
            
            void 0;
            
            if (!response.ok) {
                const errorText = await response.text();
                console.error('CarbonGuru: HTTP error response:', errorText);
                throw new Error(`HTTP ${response.status}: ${response.statusText} - ${errorText}`);
            }
            
            const data = await response.json();
            void 0;
            return data;
        } catch (error) {
            console.error('CarbonGuru: Error getting carbon footprint:', error);
            return { success: false, error: error.message };
        }
    }

    async sendToBackend(endpointOrProductData, method = 'POST') {
        try {
            // Handle both old and new calling patterns
            let endpoint, requestData, httpMethod;
            
            if (typeof endpointOrProductData === 'string') {
                // New pattern: sendToBackend(endpoint, method)
                endpoint = endpointOrProductData;
                httpMethod = method;
                requestData = null;
                void 0;
            } else {
                // Old pattern: sendToBackend(productData) - defaults to POST to analyze
                endpoint = '/api/analyze';
                httpMethod = 'POST';
                requestData = endpointOrProductData;
                void 0;
            }

            // Build fetch options (authenticatedFetch will add auth headers)
            const fetchOptions = {
                method: httpMethod
            };

            // Only include body for POST requests
            if (httpMethod === 'POST' && requestData) {
                fetchOptions.body = JSON.stringify(requestData);
            }

            // Use authenticatedFetch which handles auth headers and 401 errors automatically
            const response = await this.authenticatedFetch(`${this.backendUrl}${endpoint}`, fetchOptions);

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }
            
            const data = await response.json();
            
            if (data.success !== undefined) {
                if (data.success) {
                    void 0;
                    return data; // Return the full response object
                } else {
                    console.error(`CarbonGuru: ${httpMethod} request failed:`, data.error);
                    return { success: false, error: data.error };
                }
            } else {
                // For endpoints that don't have a success field (like mapped-items)
                void 0;
                return { success: true, ...data };
            }
        } catch (error) {
            console.error(`CarbonGuru: Error making ${method} request:`, error);
            return { success: false, error: error.message };
        }
    }

    startUpdateNotificationListener() {
        void 0;
        
        // Check for immediate updates every 30 seconds during active browsing
        this.updateCheckInterval = setInterval(async () => {
            try {
                const response = await fetch(`${this.backendUrl}/extension/notify-update`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json'
                    },
                    body: JSON.stringify({
                        currentVersion: this.currentVersion
                    })
                });
                
                if (response.ok) {
                    const data = await response.json();
                    if (data.updateAvailable && data.newVersion !== this.currentVersion) {
                        void 0;
                        this.triggerFirefoxUpdateCheck();
                    }
                }
            } catch (error) {
                // Silent fail - this is background polling
                void 0;
            }
        }, 30000); // Check every 30 seconds
        
        // Also check immediately on startup
        setTimeout(() => this.triggerFirefoxUpdateCheck(), 5000);
    }

    triggerFirefoxUpdateCheck() {
        void 0;
        
        // Firefox will check the update_url in manifest.json
        try {
            const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
            
            // In Firefox, we can only request update checks for the extension itself
            if (runtime.requestUpdateCheck) {
                runtime.requestUpdateCheck().then((status) => {
                    void 0;
                }).catch((error) => {
                    void 0;
                });
            } else {
                void 0;
            }
        } catch (error) {
            void 0;
        }
    }

    // Removed complex authentication methods - using simple hardcoded credentials
    // Removed: handleAuthenticationRequired, promptForCredentials, showAuthenticationPrompt

    async triggerAuthentication(credentials = null) {
        try {
            void 0;
            
            if (!credentials) {
                // No credentials provided - request them from popup
                void 0;
                return { success: false, needsCredentials: true };
            }
            
            // Validate credentials by making a test request
            void 0;
            const testAuthHeaders = {
                'Content-Type': 'application/json',
                'Authorization': `Basic ${btoa(`${credentials.username}:${credentials.password}`)}`
            };
            
            try {
                const testResponse = await fetch(`${this.backendUrl}/api/health`, {
                    method: 'GET',
                    headers: testAuthHeaders
                });
                
                if (!testResponse.ok) {
                    void 0;
                    return { success: false, error: 'Invalid credentials' };
                }
                
                void 0;
            } catch (error) {
                void 0;
                return { success: false, error: 'Unable to validate credentials' };
            }
            
            // Store validated credentials
            this.authCredentials = credentials;
            this.isAuthenticated = true;
            
            // Persist authentication state in storage
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            if (storage && storage.local) {
                await storage.local.set({
                    'authCredentials': credentials,
                    'isAuthenticated': true,
                    'authTimestamp': Date.now()
                });
            }
            
            void 0;
            return { success: true };
            
        } catch (error) {
            console.error('CarbonGuru: Authentication error:', error);
            return { success: false, error: error.message };
        }
    }

    // Add method to check if URL is already being processed
    isUrlBeingProcessed(url) {
        return this.processingUrls.has(url);
    }

    // Add method to mark URL as being processed
    markUrlAsProcessing(url) {
        this.processingUrls.add(url);
        void 0;
    }

    // Simple database polling for results
    async pollDatabaseForResults(process_id, url, originalTabId = null, backendAnalysisType = null, initialWcData = null) {
        void 0;
        // ✅ WOOCOMMERCE URL FIX: Store initial WooCommerce data from /api/analyze response
        if (initialWcData && initialWcData.wc_product_url) {
            void 0;
        }

        // Reset status message tracker for new polling session
        this.lastDisplayedStatusMessage = null;

        // Determine polling endpoint based on backend's analysis_type (if provided), fallback to URL detection
        // Both 'shopping_cart' and 'order_page' use cart summaries endpoint (multi-item processing)
        let isCartUrl;
        if (backendAnalysisType) {
            isCartUrl = (backendAnalysisType === 'shopping_cart' || backendAnalysisType === 'order_page');
            void 0;
        } else {
            isCartUrl = this.isShoppingCartUrl(url, '', '');
            void 0;
        }
        const pollingEndpoint = isCartUrl ? '/api/recent-cart-summaries?limit=20' : '/api/mapped-items?limit=20';

        void 0;

        let attempts = 0;
        // No timeout - poll indefinitely until results arrive or tab closes
        // This is necessary because cart processing can take 3-5+ minutes

        const pollInterval = setInterval(async () => {
            attempts++;

            // Update session poll time to track activity
            this.updateSessionPollTime(process_id);

            try {
                // Check for status messages during processing (text extraction, cleaning, etc.)
                await this.checkForStatusMessages(process_id, originalTabId);

                // Use the appropriate endpoint based on URL type
                const response = await this.sendToBackend(pollingEndpoint, 'GET');
                
                if (response.success) {
                    const items = response.items || response;

                    // Look for an item with matching process_id (successful or failed)
                    const matchingItem = items.find(item =>
                        item.session_id === process_id
                    );

                    if (matchingItem) {
                        // Check for Stage 1 (cart_extraction_status) first
                        // Stage 1 must display before any item processing begins
                        const contentType = matchingItem.content_type;
                        if (contentType === 'cart_extraction_status' &&
                            matchingItem.status_message &&
                            matchingItem.status_message !== this.lastDisplayedStatusMessage) {

                            void 0;

                            const statusMessage = {
                                action: 'analysisStatus',
                                success: true,
                                data: matchingItem
                            };

                            this.lastDisplayedStatusMessage = matchingItem.status_message;

                            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                            if (originalTabId) {
                                void 0;
                                tabs.sendMessage(originalTabId, statusMessage).catch((error) => {
                                    void 0;
                                    this.sendToActiveTab(tabs, statusMessage);
                                });
                            } else {
                                this.sendToActiveTab(tabs, statusMessage);
                            }

                            void 0;
                        }

                        // Check for Stage 2 (individual item progress)
                        if (isCartUrl && matchingItem.items && matchingItem.items.length > 0) {
                            const itemCount = matchingItem.items.length;
                            const totalItems = matchingItem.total_items || itemCount;

                            // Track last item count to detect new items
                            if (!this.lastCartItemCount) this.lastCartItemCount = {};
                            const sessionId = matchingItem.session_id;
                            const lastCount = this.lastCartItemCount[sessionId] || 0;

                            // Only show Stage 2 if:
                            // 1. We have new items (itemCount > lastCount)
                            // 2. Not all items complete yet (itemCount < totalItems)
                            // 3. Not showing completion message (content_type !== 'analysis_complete')
                            if (itemCount > lastCount && itemCount < totalItems && contentType !== 'analysis_complete') {
                                // IMPORTANT: Use current_item from ActivityLog structured_content if available
                                // This contains the correct, fresh data from the Stage 2 message
                                // Fallback to items array only if structured_content unavailable
                                let currentItemData = null;

                                if (matchingItem.structured_content && matchingItem.structured_content.current_item) {
                                    // Use current_item from ActivityLog (most accurate, fresh from backend)
                                    currentItemData = matchingItem.structured_content.current_item;
                                    void 0;
                                } else {
                                    // Fallback: Get the newest item from items array (may be stale)
                                    currentItemData = matchingItem.items[itemCount - 1];
                                    void 0;
                                }

                                const statusMessage = {
                                    action: 'analysisStatus',
                                    success: true,
                                    data: {
                                        ...matchingItem,
                                        content_type: 'individual_item_complete',
                                        current_item: currentItemData,
                                        items_complete: itemCount,
                                        total_items: totalItems
                                    }
                                };

                                // Update last count
                                this.lastCartItemCount[sessionId] = itemCount;

                                // Send to content script
                                const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

                                if (originalTabId) {
                                    void 0;
                                    tabs.sendMessage(originalTabId, statusMessage).then((response) => {
                                        void 0;
                                    }).catch((error) => {
                                        void 0;
                                        this.sendToActiveTab(tabs, statusMessage);
                                    });
                                } else {
                                    this.sendToActiveTab(tabs, statusMessage);
                                }

                                void 0;
                                return;
                            }
                        }

                        // Check for other status messages (fallback for non-cart or generic updates)
                        // Stage 1 for carts is already handled above
                        if (matchingItem.status_message && matchingItem.status_message !== this.lastDisplayedStatusMessage) {
                            void 0;

                            // For carts: Check content_type to determine stage
                            const contentType = matchingItem.content_type;
                            const isCompletionMessage = contentType === 'analysis_complete' || matchingItem.status_message.includes('✅ Analysis Complete');

                            // Send status update for non-completion messages (fallback for products or generic status)
                            if (!isCompletionMessage) {
                                // Send status message to content script immediately
                                const statusMessage = {
                                    action: 'analysisStatus',
                                    success: true,
                                    data: matchingItem
                                };

                                // Mark that we've displayed this status to avoid duplicate displays
                                this.lastDisplayedStatusMessage = matchingItem.status_message;

                                // Send to content script
                                const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

                                // Try to send to original tab first, fallback to active tab
                                if (originalTabId) {
                                    void 0;
                                    tabs.sendMessage(originalTabId, statusMessage).then((response) => {
                                        void 0;
                                    }).catch((error) => {
                                        void 0;
                                        // Fallback to active tab query
                                        this.sendToActiveTab(tabs, statusMessage);
                                    });
                                } else {
                                    void 0;
                                    this.sendToActiveTab(tabs, statusMessage);
                                }

                                // IMPORTANT: Return here to let the status message be displayed
                                // before sending the final analysisComplete message in the next poll cycle
                                void 0;
                                return;
                            }
                            // If it IS a completion message, fall through to check hasValidFootprint
                        }

                        // Check if processing was successful or failed
                        // For carts: check for completion message in status_message
                        // For products: check co2e_per_dollar
                        let hasValidFootprint = false;

                        if (isCartUrl) {
                            // Cart: Check if status_message contains "Analysis Complete"
                            // This is the backend's signal that all items are processed
                            hasValidFootprint = matchingItem.status_message &&
                                              matchingItem.status_message.includes('✅ Analysis Complete');
                        } else {
                            // Product: Check footprint and co2e_per_dollar
                            hasValidFootprint = matchingItem.product_carbon_footprint !== null &&
                                              matchingItem.product_carbon_footprint !== undefined &&
                                              matchingItem.product_carbon_footprint > 0 &&
                                              matchingItem.co2e_per_dollar !== null &&
                                              matchingItem.co2e_per_dollar > 0;
                        }

                        if (hasValidFootprint) {
                            void 0;
                        } else {
                            void 0;
                            // Continue polling - don't clear interval or return yet
                            return;
                        }

                        void 0;

                        // For products: fetch PCF comparison at display time
                        const offsetCostUsd = matchingItem.offset_cost_usd ?? matchingItem.display_cost_usd ?? matchingItem.orderable_cost_usd ?? null;
                        let productData = hasValidFootprint ? (isCartUrl ?
                            // For carts: pass through cart summary data directly (already has analysis_type, type, etc.)
                            matchingItem
                            :
                            // For products: create product data structure
                            {
                                id: matchingItem.carbonguru_id || matchingItem.id,
                                carbonguru_id: matchingItem.carbonguru_id,
                                product_carbon_footprint: matchingItem.product_carbon_footprint,
                                naics_code: matchingItem.naics_code,
                                naics_title: matchingItem.naics_title,
                                sector_code: matchingItem.sector_code,
                                sector_name: matchingItem.sector_name,
                                country_code: matchingItem.country_code,
                                product_name: matchingItem.product_name,
                                details_url: matchingItem.details_url,
                                social_cost_usd: matchingItem.social_cost_usd,
                                offset_cost_usd: offsetCostUsd,
                                display_cost_usd: matchingItem.display_cost_usd,
                                orderable_cost_usd: matchingItem.orderable_cost_usd,
                                offset_provider: matchingItem.offset_provider,
                                offset_product_id: matchingItem.offset_product_id,
                                offset_quote_source: matchingItem.offset_quote_source,
                                offset_cache_age_s: matchingItem.offset_cache_age_s,
                                offset_defaults_applied: matchingItem.offset_defaults_applied
                            }
                        ) : {
                            product_name: matchingItem.product_name,
                            error_message: 'Analysis failed during processing. This may be due to emission factor lookup issues.'
                        };

                        // Fetch PCF comparison for individual products at display time
                        if (hasValidFootprint && !isCartUrl && matchingItem.carbonguru_id) {
                            try {
                                const headers = this.getAuthHeaders();
                                const pcfResponse = await fetch(`${this.backendUrl}/api/mapped-items?carbonguru_id=${matchingItem.carbonguru_id}&limit=1`, { headers });

                                if (pcfResponse.ok) {
                                    const pcfData = await pcfResponse.json();
                                    if (pcfData.success && pcfData.items && pcfData.items.length > 0) {
                                        if (pcfData.items[0].pcf_comparison) {
                                            productData.pcf_comparison = pcfData.items[0].pcf_comparison;
                                            void 0;
                                        }
                                        // Add social cost from API response
                                        if (pcfData.items[0].social_cost_usd != null) {
                                            productData.social_cost_usd = pcfData.items[0].social_cost_usd;
                                            void 0;
                                        }
                                        // Add offset cost (prefer live value, keep existing if already set)
                                        if (pcfData.items[0].offset_cost_usd != null && productData.offset_cost_usd == null) {
                                            productData.offset_cost_usd = pcfData.items[0].offset_cost_usd;
                                            void 0;
                                        }
                                        if (pcfData.items[0].display_cost_usd != null && productData.display_cost_usd == null) {
                                            productData.display_cost_usd = pcfData.items[0].display_cost_usd;
                                        }
                                        if (pcfData.items[0].orderable_cost_usd != null && productData.orderable_cost_usd == null) {
                                            productData.orderable_cost_usd = pcfData.items[0].orderable_cost_usd;
                                        }
                                    }
                                }
                            } catch (pcfError) {
                                void 0;
                            }
                        }

                        // ✅ WOOCOMMERCE URL FIX: Include WooCommerce URL in productData if available
                        if (initialWcData && initialWcData.wc_product_url && !isCartUrl) {
                            productData.wc_product_url = initialWcData.wc_product_url;
                            productData.wc_product_id = initialWcData.wc_product_id;
                            void 0;
                        }

                        const analysisMessage = {
                            action: 'analysisComplete',
                            success: hasValidFootprint,
                            data: productData
                        };

                        // Send results to content script
                        void 0;
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

                        // Try to send to original tab first, fallback to active tab
                        // IMPORTANT: Await message delivery before cleanup to prevent Safari service worker termination
                        let messageSent = false;
                        if (originalTabId) {
                            void 0;
                            try {
                                await tabs.sendMessage(originalTabId, analysisMessage);
                                void 0;
                                messageSent = true;
                            } catch (error) {
                                void 0;
                                // Fallback to active tab query
                                messageSent = await this.sendToActiveTabAsync(tabs, analysisMessage);
                            }
                        } else {
                            void 0;
                            messageSent = await this.sendToActiveTabAsync(tabs, analysisMessage);
                        }

                        // Also send to popup (runtime message for popup to listen to)
                        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        try {
                            await runtime.sendMessage(analysisMessage);
                        } catch (error) {
                            // Popup might not be open - this is expected behavior
                            void 0;
                        }

                        // Small delay to ensure message is processed before cleanup (Safari service worker fix)
                        await new Promise(resolve => setTimeout(resolve, 100));

                        // Clean up polling now that completion message has been sent
                        clearInterval(pollInterval);
                        this.removeSession(process_id);
                        this.removeUrlFromProcessing(url);
                        // Reset status message tracker for next analysis
                        this.lastDisplayedStatusMessage = null;
                        return;
                    }
                }

                // No timeout check - continue polling until results arrive
                // Results arrival triggers cleanup at line ~2459-2463
            } catch (error) {
                console.error('CarbonGuru: Polling error:', error);

                // Handle 404 or failed sessions gracefully
                if (error.message && (error.message.includes('404') || error.message.includes('failed'))) {
                    void 0;
                    this.removeSession(process_id);
                    clearInterval(pollInterval);
                    this.removeUrlFromProcessing(url);
                    return;
                }
                // Don't stop polling on other errors - cart processing may still complete
            }
        }, 1000); // Poll every 1 second for minimal perceived latency during processing

        // Track this session for management
        this.trackSession(process_id, pollInterval);
    }

    // Check for status messages during processing (text extraction, cleaning, etc.)
    async checkForStatusMessages(process_id, originalTabId) {
        try {
            const response = await this.sendToBackend(`/api/activity-status?session_id=${process_id}`, 'GET');

            if (response.success && response.status_messages && response.status_messages.length > 0) {
                // Initialize displayed messages set if not exists
                if (!this.displayedStatusMessages) {
                    this.displayedStatusMessages = new Set();
                }

                // Process new status messages
                for (const statusMessage of response.status_messages) {
                    // Skip if we've already displayed this message
                    if (this.displayedStatusMessages.has(statusMessage.id)) {
                        continue;
                    }

                    const contentType = statusMessage.content_type;
                    if (contentType === 'text_extraction_status' || contentType === 'text_cleaning_status' ||
                        contentType === 'cart_extraction_status' || contentType === 'individual_item_complete' ||
                        contentType === 'analysis_complete' || contentType === 'product_extraction_status' ||
                        contentType === 'learning_triggered_status') {
                        void 0;

                        let messageData;
                        if (contentType === 'analysis_complete') {
                            // Analysis complete should trigger analysisComplete action
                            // CRITICAL: Extract analysis_type from structured_content and add to top level
                            // This matches what we do for cart_extraction_status and individual_item_complete
                            const completionData = statusMessage.structured_content || {
                                content_type: contentType,
                                message: statusMessage.message,
                                timestamp: statusMessage.timestamp
                            };

                            // ALWAYS set content_type (required by content.js early return check)
                            // Backend stores content_type in ActivityLog.content_type, not in structured_content
                            completionData.content_type = contentType;

                            // Extract and promote analysis_type to top level (required by content.js routing)
                            // Backend stores analysis_type inside structured_content, but content.js expects it at message.data.analysis_type
                            if (statusMessage.structured_content && statusMessage.structured_content.analysis_type) {
                                completionData.analysis_type = statusMessage.structured_content.analysis_type;
                                // Set type for legacy compatibility
                                if (completionData.analysis_type === 'order_page') {
                                    completionData.type = 'order';
                                } else if (completionData.analysis_type === 'individual_product') {
                                    completionData.type = 'product';
                                } else {
                                    completionData.type = 'cart';
                                }
                            } else {
                                // Fallback for legacy carts without analysis_type
                                completionData.analysis_type = 'shopping_cart';
                                completionData.type = 'cart';
                            }

                            messageData = {
                                action: 'analysisComplete',
                                success: true,
                                data: completionData
                            };
                        } else {
                            // Other status messages use analysisStatus action
                            const baseData = {
                                content_type: contentType,
                                structured_content: statusMessage.structured_content,
                                message: statusMessage.message,
                                timestamp: statusMessage.timestamp
                            };

                            // For cart-specific messages, add additional fields expected by content script
                            if (contentType === 'cart_extraction_status' || contentType === 'individual_item_complete' || contentType === 'learning_triggered_status') {
                                // Preserve analysis_type from backend (order_page or shopping_cart)
                                // Only set if not already present in structured_content
                                if (statusMessage.structured_content && statusMessage.structured_content.analysis_type) {
                                    baseData.analysis_type = statusMessage.structured_content.analysis_type;
                                } else {
                                    baseData.analysis_type = 'shopping_cart';  // Fallback for legacy carts
                                }
                                // Set type for legacy compatibility
                                if (baseData.analysis_type === 'order_page') {
                                    baseData.type = 'order';
                                } else {
                                    baseData.type = 'cart';
                                }

                                // Parse structured content for cart data
                                if (statusMessage.structured_content) {
                                    if (statusMessage.structured_content.items) {
                                        baseData.items = statusMessage.structured_content.items;
                                    }
                                    // Backend sends 'item_count' in cart_extraction_status structured_content
                                    // Map it to 'total_items' for content.js
                                    if (statusMessage.structured_content.item_count) {
                                        baseData.total_items = statusMessage.structured_content.item_count;
                                    } else if (statusMessage.structured_content.total_items) {
                                        baseData.total_items = statusMessage.structured_content.total_items;
                                    }
                                    // Backend sends 'enhanced_title' in ActivityLog structured_content
                                    // Map it to 'enhanced_display_title' for content.js
                                    if (statusMessage.structured_content.enhanced_title) {
                                        baseData.enhanced_display_title = statusMessage.structured_content.enhanced_title;
                                        baseData.product_name = statusMessage.structured_content.enhanced_title;
                                    } else if (statusMessage.structured_content.enhanced_display_title) {
                                        // Fallback for other sources that might use enhanced_display_title
                                        baseData.enhanced_display_title = statusMessage.structured_content.enhanced_display_title;
                                        baseData.product_name = statusMessage.structured_content.enhanced_display_title;
                                    }
                                    // For individual_item_complete: pass through current_item and items_complete
                                    if (contentType === 'individual_item_complete') {
                                        if (statusMessage.structured_content.current_item) {
                                            baseData.current_item = statusMessage.structured_content.current_item;
                                        }
                                        if (statusMessage.structured_content.items_complete !== undefined) {
                                            baseData.items_complete = statusMessage.structured_content.items_complete;
                                        }
                                    }
                                }
                            }

                            messageData = {
                                action: 'analysisStatus',
                                success: true,
                                data: baseData
                            };
                        }

                        // Send to content script
                        // For analysis_complete: delay 2s so Stage 2 (individual item progress) is visible
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        if (contentType === 'analysis_complete') {
                            void 0;
                            setTimeout(() => {
                                if (originalTabId) {
                                    tabs.sendMessage(originalTabId, messageData).catch(() => {
                                        this.sendToActiveTab(tabs, messageData);
                                    });
                                } else {
                                    this.sendToActiveTab(tabs, messageData);
                                }
                            }, 2000);
                        } else if (originalTabId) {
                            try {
                                await tabs.sendMessage(originalTabId, messageData);
                                void 0;
                            } catch (error) {
                                void 0;
                                this.sendToActiveTab(tabs, messageData);
                            }
                        } else {
                            this.sendToActiveTab(tabs, messageData);
                        }

                        // Mark message as displayed ONLY if it's complete/final
                        // Don't mark 'pending' status messages as displayed - backend updates them with real data (same ID)
                        const messageStatus = statusMessage.structured_content?.status;
                        if (messageStatus !== 'pending') {
                            this.displayedStatusMessages.add(statusMessage.id);
                            void 0;
                        } else {
                            void 0;
                        }
                    }
                }
            }
        } catch (error) {
            // Silent - status message polling is supplementary, don't log errors unless debugging
            // void 0;
        }
    }

    // Helper function to send message to active tab as fallback
    sendToActiveTab(tabs, analysisMessage) {
        tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
            void 0;
            if (tabsArray[0]) {
                void 0;
                tabs.sendMessage(tabsArray[0].id, analysisMessage).then(() => {
                    void 0;
                }).catch((error) => {
                    void 0;
                });
            } else {
                void 0;
            }
        }).catch((error) => {
            void 0;
        });
    }

    // Async version of sendToActiveTab that returns a promise (for Safari service worker reliability)
    async sendToActiveTabAsync(tabs, analysisMessage) {
        try {
            const tabsArray = await tabs.query({ active: true, currentWindow: true });
            void 0;
            if (tabsArray[0]) {
                void 0;
                await tabs.sendMessage(tabsArray[0].id, analysisMessage);
                void 0;
                return true;
            } else {
                void 0;
                return false;
            }
        } catch (error) {
            void 0;
            return false;
        }
    }

    // Gmail email processing - unified with cart/product async flow
    async processGmailEmail(messageId, tabId) {
        void 0;

        try {
            // Call backend API to initiate async email processing
            const response = await this.authenticatedFetch(`${this.backendUrl}/api/gmail/process-emails`, {
                method: 'POST',
                body: JSON.stringify({
                    message_id: messageId,
                    single_message: true
                })
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
            }

            const result = await response.json();
            void 0;

            if (!result.success) {
                throw new Error(result.message || result.error || 'Email processing failed');
            }

            // Backend returns process_id for async tracking
            const processId = result.process_id;
            void 0;

            // Validate processId exists before polling
            if (!processId) {
                console.error('CarbonGuru Gmail: Backend returned success but no process_id!', result);
                throw new Error('Backend did not return a valid process_id');
            }

            // Start polling using SAME mechanism as shopping carts (not Gmail-specific polling)
            // Gmail sessions have analysis_type='shopping_cart', so they appear in /api/recent-cart-summaries
            // This ensures Stage 1 (cart extraction status), Stage 2 (item progress), and Stage 3 (complete) all display correctly
            void 0;
            this.pollDatabaseForResults(
                processId,                         // process_id
                `gmail://message/${processId}`,    // url - use gmail:// scheme
                tabId,                             // originalTabId
                'shopping_cart',                   // backendAnalysisType - emails are processed as carts
                null                               // initialWcData (not applicable for emails)
            );

            return {
                success: true,
                message: 'Email processing started',
                process_id: processId
            };

        } catch (error) {
            console.error('CarbonGuru Gmail: Error initiating email processing:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    // Process Gmail emails by date range (for all scope options: recent, category, date-based)
    async processGmailDateRange(options, tabId) {
        void 0;

        try {
            // Call backend API to initiate async email processing
            // Options: { days_back, limit, category, document_types }
            const response = await this.authenticatedFetch(`${this.backendUrl}/api/gmail/process-emails`, {
                method: 'POST',
                body: JSON.stringify(options)
            });

            if (!response.ok) {
                const errorData = await response.json();
                throw new Error(errorData.message || `HTTP ${response.status}: ${response.statusText}`);
            }

            const result = await response.json();
            void 0;

            if (!result.success) {
                throw new Error(result.message || result.error || 'Email processing failed');
            }

            // Backend returns process_id for async tracking
            const processId = result.process_id;
            void 0;

            // Validate processId exists before polling
            if (!processId) {
                console.error('CarbonGuru Gmail: Backend returned success but no process_id!', result);
                throw new Error('Backend did not return a valid process_id');
            }

            // Start polling using SAME mechanism as shopping carts
            void 0;
            this.pollDatabaseForResults(
                processId,                         // process_id
                `gmail://range/${processId}`,      // url - use gmail:// scheme
                tabId,                             // originalTabId
                'shopping_cart',                   // backendAnalysisType - emails are processed as carts
                null                               // initialWcData (not applicable for emails)
            );

            return {
                success: true,
                message: 'Email processing started',
                process_id: processId
            };

        } catch (error) {
            console.error('CarbonGuru Gmail: Error initiating email processing:', error);
            return {
                success: false,
                error: error.message
            };
        }
    }

    // Gmail polling for async email processing - uses EXACT same pattern as cart polling
    async pollGmailResults(processId, tabId) {
        void 0;

        let attempts = 0;
        const maxAttempts = 120; // 2 minutes (120 * 1 second)

        const pollInterval = setInterval(async () => {
            attempts++;

            // Update session poll time to track activity
            this.updateSessionPollTime(processId);

            try {
                // CRITICAL: Use checkForStatusMessages() SAME as cart polling (line 2516)
                // This handles ALL activity status messages (Stage 1, 2, 3) via /api/activity-status
                await this.checkForStatusMessages(processId, tabId);

                // CRITICAL: Poll /api/recent-cart-summaries SAME as cart polling (line 2519)
                // Gmail sessions have analysis_type='shopping_cart' so they appear in this endpoint!
                const response = await this.sendToBackend('/api/recent-cart-summaries?limit=20', 'GET');

                if (response.success) {
                    const items = response.items || response;

                    // Look for an item with matching process_id (session_id)
                    const matchingItem = items.find(item =>
                        item.session_id === processId
                    );

                    if (matchingItem) {
                        // Check for Stage 1 (cart_extraction_status) first
                        const contentType = matchingItem.content_type;
                        if (contentType === 'cart_extraction_status' &&
                            matchingItem.status_message &&
                            matchingItem.status_message !== this.lastDisplayedStatusMessage) {

                            void 0;

                            const statusMessage = {
                                action: 'analysisStatus',
                                success: true,
                                data: matchingItem
                            };

                            this.lastDisplayedStatusMessage = matchingItem.status_message;

                            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                            try {
                                await tabs.sendMessage(tabId, statusMessage);
                                void 0;
                            } catch (error) {
                                void 0;
                            }

                            void 0;
                        }

                        // Check for Stage 2 (individual item progress)
                        if (matchingItem.items && matchingItem.items.length > 0) {
                            const itemCount = matchingItem.items.length;
                            const totalItems = matchingItem.total_items || itemCount;

                            // Track last item count to detect new items
                            if (!this.lastCartItemCount) this.lastCartItemCount = {};
                            const sessionId = matchingItem.session_id;
                            const lastCount = this.lastCartItemCount[sessionId] || 0;

                            // Only show Stage 2 if we have new items and not all complete
                            if (itemCount > lastCount && itemCount < totalItems && contentType !== 'analysis_complete') {
                                let currentItemData = null;

                                if (matchingItem.structured_content && matchingItem.structured_content.current_item) {
                                    // Use current_item from ActivityLog (most accurate)
                                    currentItemData = matchingItem.structured_content.current_item;
                                    void 0;
                                } else {
                                    // Fallback: Get the newest item from items array
                                    currentItemData = matchingItem.items[itemCount - 1];
                                    void 0;
                                }

                                void 0;

                                const progressMessage = {
                                    action: 'analysisStatus',
                                    success: true,
                                    data: {
                                        content_type: 'individual_item_complete',
                                        analysis_type: matchingItem.analysis_type || 'shopping_cart',  // Preserve actual type from backend
                                        type: matchingItem.type || 'cart',  // Preserve actual type from backend
                                        session_id: sessionId,
                                        current_item: currentItemData,
                                        items_complete: itemCount,
                                        total_items: totalItems,
                                        message: `Processing item ${itemCount} of ${totalItems}: ${currentItemData.product_name}`
                                    }
                                };

                                this.lastCartItemCount[sessionId] = itemCount;

                                const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                                try {
                                    await tabs.sendMessage(tabId, progressMessage);
                                    void 0;
                                } catch (error) {
                                    void 0;
                                }
                            }
                        }

                        // Check for Stage 3 (analysis_complete) via content_type
                        if (contentType === 'analysis_complete') {
                            void 0;
                            clearInterval(pollInterval);

                            const completionMessage = {
                                action: 'analysisComplete',
                                success: true,
                                data: matchingItem
                            };

                            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                            try {
                                await tabs.sendMessage(tabId, completionMessage);
                                void 0;
                            } catch (error) {
                                void 0;
                            }

                            return;
                        }
                    }
                }

                // Fallback: Check /api/gmail/status for completion (in case session not in recent-cart-summaries yet)
                // This is a safety net - primary completion detection is via content_type above
                const statusResponse = await this.authenticatedFetch(
                    `${this.backendUrl}/api/gmail/status/${processId}`,
                    { method: 'GET' }
                );

                if (statusResponse.ok) {
                    const asyncData = await statusResponse.json();

                    // Only use fallback if we haven't found the session in recent-cart-summaries yet
                    if (asyncData.status === 'error' || asyncData.status === 'failed') {
                        console.error('CarbonGuru Gmail: Processing failed:', asyncData.error);
                        clearInterval(pollInterval);

                        const errorMessage = {
                            action: 'analysisComplete',
                            success: false,
                            data: {
                                error_message: asyncData.error || 'Email processing failed'
                            }
                        };

                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        try {
                            await tabs.sendMessage(tabId, errorMessage);
                        } catch (error) {
                            void 0;
                        }

                        return;
                    }
                }

                // Check timeout
                if (attempts >= maxAttempts) {
                    console.error('CarbonGuru Gmail: Polling timeout');
                    clearInterval(pollInterval);

                    const timeoutMessage = {
                        action: 'analysisComplete',
                        success: false,
                        data: {
                            error_message: 'Email processing timeout - please try again'
                        }
                    };

                    const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                    try {
                        await tabs.sendMessage(tabId, timeoutMessage);
                    } catch (error) {
                        void 0;
                    }
                }

            } catch (error) {
                console.error('CarbonGuru Gmail: Polling error:', error);
            }

        }, 1000); // Poll every 1 second
    }

    // Cart polling for async cart processing
    async pollCartResults(cart_id, url, polling_url) {
        void 0;
        
        let attempts = 0;
        const maxAttempts = 120; // Poll for 4 minutes (120 * 2 seconds) - longer for carts
        
        const pollInterval = setInterval(async () => {
            attempts++;
            
            try {
                // Poll the cart status endpoint
                const response = await fetch(`${this.backendUrl}${polling_url}`, {
                    method: 'GET',
                    headers: this.getAuthHeaders()
                });
                
                if (response.ok) {
                    const cartData = await response.json();
                    
                    if (cartData.status === 'completed' && cartData.success) {
                        // Cart processing completed successfully
                        void 0;
                        clearInterval(pollInterval);
                        this.removeUrlFromProcessing(url);
                        
                        // Send cart results to content script
                        const analysisMessage = {
                            action: 'analysisComplete',
                            success: true,
                            data: {
                                analysis_type: cartData.analysis_type || 'shopping_cart',  // Preserve actual type from backend
                                type: cartData.type || 'cart',  // Preserve actual type from backend
                                cart_id: cart_id,
                                total_footprint: cartData.total_footprint || 0,
                                item_count: cartData.item_count || 0,
                                items: cartData.items || [],
                                processing_stats: cartData.processing_stats || {},
                                details_url: cartData.details_url || `/sessions/${cartData.session_id}`,
                                message: `Cart analysis complete: ${cartData.item_count || 0} items processed`
                            }
                        };
                        
                        // Send to content script
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                            if (tabsArray[0]) {
                                tabs.sendMessage(tabsArray[0].id, analysisMessage).catch((error) => {
                                    void 0;
                                });
                            }
                        });
                        
                        // Also send to popup
                        const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        runtime.sendMessage(analysisMessage).catch((error) => {
                            void 0;
                        });
                        
                        return;
                        
                    } else if (cartData.status === 'failed') {
                        // Cart processing failed
                        console.error('CarbonGuru: Cart processing failed:', cartData.error);
                        clearInterval(pollInterval);
                        this.removeUrlFromProcessing(url);
                        
                        // Send error message to content script
                        const errorMessage = {
                            action: 'analysisComplete',
                            success: false,
                            data: {
                                error_message: cartData.error || 'Cart processing failed',
                                cart_id: cart_id
                            }
                        };
                        
                        const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                        tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                            if (tabsArray[0]) {
                                tabs.sendMessage(tabsArray[0].id, errorMessage).catch((error) => {
                                    void 0;
                                });
                            }
                        });
                        
                        return;
                        
                    } else if (cartData.status === 'processing') {
                        // Still processing - update progress if available
                        if (cartData.progress_percent !== undefined) {
                            void 0;
                            
                            // Send progress update to content script
                            const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                            tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                                if (tabsArray[0]) {
                                    tabs.sendMessage(tabsArray[0].id, {
                                        action: 'updateProgress',
                                        progress: cartData.progress_percent,
                                        message: `Processing cart... ${cartData.progress_percent}% complete`,
                                        estimatedRemaining: cartData.estimated_remaining_seconds
                                    }).catch((error) => {
                                        void 0;
                                    });
                                }
                            });
                        }
                        // Continue polling
                        return;
                    }
                }
                
                if (attempts >= maxAttempts) {
                    void 0;
                    clearInterval(pollInterval);
                    this.removeUrlFromProcessing(url);
                    
                    // Send timeout message
                    const timeoutMessage = {
                        action: 'analysisComplete',
                        success: false,
                        data: {
                            error_message: 'Cart processing timed out. Please try again.',
                            cart_id: cart_id
                        }
                    };
                    
                    const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                    tabs.query({ active: true, currentWindow: true }).then((tabsArray) => {
                        if (tabsArray[0]) {
                            tabs.sendMessage(tabsArray[0].id, timeoutMessage).catch((error) => {
                                void 0;
                            });
                        }
                    });
                }
                
            } catch (error) {
                console.error('CarbonGuru: Cart polling error:', error);
                if (attempts >= maxAttempts) {
                    clearInterval(pollInterval);
                    this.removeUrlFromProcessing(url);
                }
            }
        }, 1000); // Poll every 1 second for minimal perceived latency during processing
    }

    // Add method to remove URL from processing
    removeUrlFromProcessing(url) {
        this.processingUrls.delete(url);
        void 0;
    }

    // Add method to load authentication state from storage
    async loadAuthenticationState() {
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            if (storage && storage.local) {
                const result = await storage.local.get(['authCredentials', 'isAuthenticated', 'authTimestamp']);
                
                if (result.authCredentials && result.isAuthenticated) {
                    // Check if authentication is still valid (within 24 hours)
                    const authAge = Date.now() - (result.authTimestamp || 0);
                    const maxAuthAge = 24 * 60 * 60 * 1000; // 24 hours
                    
                    if (authAge < maxAuthAge) {
                        this.authCredentials = result.authCredentials;
                        this.isAuthenticated = true;
                        void 0;
                        return true;
                    } else {
                        void 0;
                        await this.clearAuthenticationState();
                    }
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Error loading authentication state:', error);
        }
        return false;
    }

    // Add method to clear authentication state
    async clearAuthenticationState() {
        this.authCredentials = null;
        this.isAuthenticated = false;

        const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
        if (storage && storage.local) {
            await storage.local.remove(['authCredentials', 'isAuthenticated', 'authTimestamp']);
        }

        void 0;
    }

    // Get current authentication status with user info
    async getAuthStatus(bustCache = false) {
        try {
            void 0;

            // Force reload token from storage if cache busting requested
            if (bustCache) {
                void 0;
                await this.oauthManager.loadStoredToken();
            }

            // Debug OAuth manager state
            void 0;

            const isAuthenticated = this.oauthManager.isAuthenticated();
            void 0;

            if (!isAuthenticated) {
                void 0;
                return {
                    isAuthenticated: false,
                    user: null,
                    reason: 'No valid JWT token or token expired'
                };
            }

            // Try to get user info from stored JWT token
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            const result = await storage.local.get(['user_info']);

            void 0;

            const authStatus = {
                isAuthenticated: true,
                user: result.user_info || { email: 'Authenticated User' },
                tokenExpiry: this.oauthManager.tokenExpiry ? new Date(this.oauthManager.tokenExpiry).toISOString() : null,
                backendUrl: this.backendUrl
            };

            void 0;
            return authStatus;

        } catch (error) {
            console.error('CarbonGuru: Error getting auth status:', error);
            return {
                isAuthenticated: false,
                user: null,
                reason: `Error: ${error.message}`
            };
        }
    }

    /**
     * Check Gmail connection status via backend API
     * @returns {Promise<Object>} Gmail connection status with email if connected
     */
    async checkGmailConnectionStatus() {
        try {
            void 0;

            // First check if user is authenticated
            const authHeader = this.oauthManager.getAuthHeader();
            if (!authHeader) {
                void 0;
                return {
                    connected: false,
                    email: null,
                    error: 'Not authenticated with CarbonGuru',
                    backendUrl: this.backendUrl
                };
            }

            // Call backend Gmail status API
            const response = await fetch(`${this.backendUrl}/api/gmail/status`, {
                method: 'GET',
                headers: {
                    ...authHeader
                }
            });

            void 0;

            if (!response.ok) {
                console.error('CarbonGuru Background: Gmail status check failed:', response.status);
                return {
                    connected: false,
                    email: null,
                    error: `HTTP ${response.status}`,
                    backendUrl: this.backendUrl
                };
            }

            const result = await response.json();
            void 0;

            return {
                connected: result.connected || false,
                email: result.email || null,
                error: result.error || null,
                backendUrl: this.backendUrl
            };

        } catch (error) {
            console.error('CarbonGuru Background: Gmail status check error:', error);
            return {
                connected: false,
                email: null,
                error: error.message,
                backendUrl: this.backendUrl
            };
        }
    }

    // Set up OAuth callback URL listener
    setupOAuthCallbackListener() {
        void 0;

        const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;

        // Listen for tab updates to intercept OAuth callback
        tabsAPI.onUpdated.addListener(async (tabId, changeInfo, tab) => {
            // Log all OAuth-related tab updates for debugging
            if (tab.url && (tab.url.includes('/oauth/') || tab.url.includes('/extension/oauth/'))) {
                void 0;
            }

            // Only process complete page loads
            if (changeInfo.status !== 'complete' || !tab.url) {
                return;
            }

            // Check if this is our OAuth callback URL
            // Use pattern matching that works even if service worker restarted and this.backendUrl is not yet set
            const isOAuthCallback = tab.url.includes('carbonguru.io/extension/oauth/callback') ||
                                   tab.url.includes('carbonguru.io/extension/oauth/callback');

            // Also try the instance-based URL if available
            const callbackUrl = this.backendUrl ? `${this.backendUrl}/extension/oauth/callback` : null;
            const matchesInstanceUrl = callbackUrl && tab.url.startsWith(callbackUrl);

            void 0;

            if (!isOAuthCallback && !matchesInstanceUrl) {
                return;
            }

            // Ensure backendUrl is set for token exchange (derive from callback URL if needed)
            if (!this.backendUrl) {
                // Extract backend URL from the callback URL
                const urlObj = new URL(tab.url);
                this.backendUrl = `${urlObj.protocol}//${urlObj.host}`;
                void 0;

                // Also reinitialize OAuth manager with correct backend
                if (this.oauthManager) {
                    await this.oauthManager.init({ backendUrl: this.backendUrl });
                }
            }

            void 0;

            try {
                // Parse URL to extract authorization code
                const url = new URL(tab.url);
                const code = url.searchParams.get('code');
                const state = url.searchParams.get('state');
                const error = url.searchParams.get('error');

                if (error) {
                    console.error('CarbonGuru OAuth: Authorization error:', error);
                    // Close the tab
                    tabsAPI.remove(tabId);
                    return;
                }

                if (!code) {
                    console.error('CarbonGuru OAuth: No authorization code in callback URL');
                    return;
                }

                void 0;

                // Handle the callback - exchange code for JWT token
                const result = await this.oauthManager.handleCallback(code, state);

                if (result.success) {
                    void 0;

                    // Reload token from storage into memory
                    await this.oauthManager.loadStoredToken();
                    void 0;

                    // Close the authorization tab
                    tabsAPI.remove(tabId);

                    // Show success notification
                    const notificationsAPI = typeof browser !== 'undefined' ? browser.notifications : chrome.notifications;
                    if (notificationsAPI) {
                        const runtimeAPI = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        notificationsAPI.create({
                            type: 'basic',
                            iconUrl: runtimeAPI.getURL('icons/icon48.png'),
                            title: 'CarbonGuru Authentication',
                            message: `Successfully signed in as ${result.user?.email || result.user?.display_name || 'user'}!`
                        });
                    }

                    // Notify popup to refresh auth status
                    try {
                        const runtimeAPI = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        runtimeAPI.sendMessage({
                            action: 'authenticationStatusChanged',
                            isAuthenticated: true,
                            user: result.user
                        }).catch(() => {
                            // Popup may not be open - that's fine
                        });
                    } catch (e) {
                        // Ignore - popup may not be open
                    }

                    // Notify all content scripts about auth change
                    try {
                        const tabs = await tabsAPI.query({});
                        for (const tab of tabs) {
                            if (tab.id && tab.url && !tab.url.startsWith('chrome://') && !tab.url.startsWith('about:')) {
                                tabsAPI.sendMessage(tab.id, {
                                    action: 'authenticationStatusChanged',
                                    isAuthenticated: true,
                                    user: result.user
                                }).catch(() => {
                                    // Tab may not have content script - ignore
                                });
                            }
                        }
                    } catch (e) {
                        void 0;
                    }
                } else {
                    console.error('CarbonGuru OAuth: Token exchange failed:', result.error);

                    // Show error notification
                    const notificationsAPI = typeof browser !== 'undefined' ? browser.notifications : chrome.notifications;
                    if (notificationsAPI) {
                        const runtimeAPI = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
                        notificationsAPI.create({
                            type: 'basic',
                            iconUrl: runtimeAPI.getURL('icons/icon48.png'),
                            title: 'CarbonGuru Authentication Failed',
                            message: result.error || 'Failed to complete authentication'
                        });
                    }

                    // Close the tab after a delay to let user see any error
                    setTimeout(() => {
                        tabsAPI.remove(tabId);
                    }, 3000);
                }

            } catch (error) {
                console.error('CarbonGuru OAuth: Error processing callback:', error);

                // Close the tab
                setTimeout(() => {
                    tabsAPI.remove(tabId);
                }, 3000);
            }
        });

        void 0;
    }


    // Add method to get current version info
    getVersionInfo() {
        const manifest = browser.runtime.getManifest();
        const versionInfo = {
            currentVersion: this.currentVersion,
            extensionVersion: this.extensionVersion,
            buildNumber: manifest.build_number || null,  // Safari build number (full 4th component)
            isDevelopment: this.isDevelopment,
            backendUrl: this.backendUrl
        };
        void 0;
        return versionInfo;
    }

    async getRecentItems(limit = 10) {
        try {
            void 0;

            // Fetch both individual items and cart summaries in parallel using authenticated fetch
            const [itemsResponse, cartsResponse] = await Promise.all([
                this.authenticatedFetch(`${this.backendUrl}/api/mapped-items?limit=${limit}`),
                this.authenticatedFetch(`${this.backendUrl}/api/recent-cart-summaries?limit=${limit}`)
            ]);

            let allItems = [];

            // Process individual items
            if (itemsResponse.ok) {
                const itemsData = await itemsResponse.json();
                if (itemsData.success && itemsData.items) {
                    void 0;
                    // Mark individual items with type
                    const individualItems = itemsData.items.map(item => ({
                        ...item,
                        item_type: 'individual'
                    }));
                    allItems = allItems.concat(individualItems);
                }
            } else {
                void 0;
            }

            // Process cart summaries
            if (cartsResponse.ok) {
                const cartsData = await cartsResponse.json();
                if (cartsData.success && cartsData.items) {
                    void 0;
                    // Mark cart summaries with type
                    const cartItems = cartsData.items.map(item => ({
                        ...item,
                        item_type: 'cart'
                    }));
                    allItems = allItems.concat(cartItems);
                }
            } else {
                void 0;
            }

            // Sort all items by date (newest first)
            allItems.sort((a, b) => {
                const dateA = new Date(a.date_created || a.created_at || 0);
                const dateB = new Date(b.date_created || b.created_at || 0);
                return dateB - dateA;
            });

            // Limit to requested number of items
            const limitedItems = allItems.slice(0, limit);

            void 0;

            return {
                success: true,
                items: limitedItems
            };
        } catch (error) {
            console.error('CarbonGuru: Error getting recent items:', error);
            return {
                success: false,
                error: error.message,
                items: []
            };
        }
    }

    /**
     * Process shopping cart data using recognition-first approach
     * Phase 1 implementation - sends raw cart data to backend
     */
    async processRawContent(contentData, sendResponse) {
        try {
            void 0;
            
            // Validate content data
            if (!contentData.url || !contentData.full_page_content) {
                throw new Error('Invalid content data: missing URL or HTML content');
            }
            
            void 0;
            
            // Send to unified backend API - let backend decide cart vs product
            const apiUrl = `${this.getApiBaseUrl()}/api/analyze`;
            const requestData = {
                url: contentData.url,
                full_page_content: contentData.full_page_content,
                page_title: contentData.page_title,
                extracted_at: contentData.extracted_at,
                extraction_method: contentData.extraction_method,
                source: 'browser_extension',
                version: SCRIPT_VERSION
            };
            
            const response = await fetch(apiUrl, {
                method: 'POST',
                headers: {
                    ...this.getAuthHeaders(),
                    'X-Extension-Version': SCRIPT_VERSION
                },
                body: JSON.stringify(requestData)
            });
            
            if (!response.ok) {
                const errorText = await response.text();
                throw new Error(`API request failed: ${response.status} - ${errorText}`);
            }
            
            const result = await response.json();
            
            // Handle async processing response
            if (result.status === 'processing_async' && result.process_id && result.polling_url) {
                void 0;
                
                // Send progress update to content script
                try {
                    const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                    const tabsArray = await tabs.query({ active: true, currentWindow: true });
                    if (tabsArray[0]) {
                        await tabs.sendMessage(tabsArray[0].id, {
                            action: 'showUnifiedNotification',
                            message: 'Processing carbon footprint analysis...',
                            options: { 
                                type: 'progress', 
                                showProgress: true
                            }
                        });
                    }
                } catch (messageError) {
                    void 0;
                }
                
                // Start polling for results
                this.pollAnalysisResults(result.process_id, result.polling_url, contentData.url);
                
                // Send immediate response that processing started
                sendResponse({
                    success: true,
                    status: 'processing_async',
                    process_id: result.process_id,
                    message: 'Analysis started - polling for results'
                });
                
            } else {
                // Handle synchronous response (legacy)
                void 0;
                
                // Store processed items in recent items
                if (result.success) {
                    if (result.type === 'cart' && result.items) {
                        // Store all cart items
                        for (const item of result.items) {
                            await this.storeRecentItem(item);
                        }
                    } else if (result.type === 'product') {
                        // Store single product
                        await this.storeRecentItem(result);
                    }
                }
                
                // Send analysisComplete message to content script to update floating tile
                try {
                    const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                    const tabsArray = await tabs.query({ active: true, currentWindow: true });
                    if (tabsArray[0]) {
                        await tabs.sendMessage(tabsArray[0].id, {
                            action: 'analysisComplete',
                            success: result.success,
                            data: result
                        });
                        void 0;
                    }
                } catch (messageError) {
                    void 0;
                }
                
                sendResponse(result);
            }
            
        } catch (error) {
            console.error('🔍 CarbonGuru: Content processing error:', error);
            sendResponse({
                success: false,
                error: error.message,
                type: 'unknown'
            });
        }
    }
    
    /**
     * Check if a URL is a shopping cart based on URL patterns, page title, and content
     */
    isShoppingCartUrl(url, pageTitle = '', pageContent = '') {
        try {
            // FIRST: Exclude product pages - they should never be classified as carts
            // even if they contain cart-related content (e.g., "Add to Cart" buttons)
            if (this.isProductPage(url)) {
                void 0;
                return false;
            }

            // SECOND: Exclude order pages - they should never be classified as carts
            // Order pages often contain cart-like content but are fundamentally different
            const orderExclusionPatterns = [
                /\/your-orders?\b/i,                            // Amazon: /your-orders/
                /\/order-?details?\b/i,                         // Amazon: /order-details
                /\/order-?history\b/i,                          // Order history pages
                /\/order-?status\b/i,                           // Order status pages
                /\/order-?tracking\b/i,                         // Order tracking pages
                /\/orders?\/\d+/i,                              // Order ID in path: /order/123456
                /\/purchases?\b/i,                              // Purchase history
                /\/account\/orders?\b/i,                        // Account order pages
                /\/my-?orders?\b/i,                             // My orders pages
                /orderid=/i,                                    // Order ID in query string
                /order_id=/i,                                   // Order ID in query string
                /\/receipt\b/i,                                 // Receipt pages
                /\/invoice\b/i,                                 // Invoice pages
                /\/confirmation\b/i                             // Order confirmation pages
            ];

            const urlLower = url.toLowerCase();
            for (const pattern of orderExclusionPatterns) {
                if (pattern.test(urlLower)) {
                    void 0;
                    return false;
                }
            }

            // URL-based cart detection patterns
            const cartUrlPatterns = [
                /\/carts?\b/i,                              // cart or carts
                /\/baskets?\b/i,                            // basket or baskets
                /\/shopping-?carts?\b/i,                    // shopping-cart or shopping-carts
                /\/checkouts?\b/i,                          // checkout or checkouts (Newbury Comics fix)
                /\/bags?\b/i,                               // bag or bags
                /\/order-?reviews?\b/i,                     // order-review or order-reviews
                /\/shoppingcarts?\b/i,                      // shoppingcart or shoppingcarts
                /\/mycarts?\b/i,                            // mycart or mycarts
                /\/viewcarts?\b/i,                          // viewcart (Uline uses /Product/ViewCart)
                /\/commande\b/i,                            // commande (French: order) - AlpinStore
                /\/panier\b/i,                              // panier (French: basket)
                /\/chariot\b/i,                             // chariot (French: cart)
                /carts?\.(html?|cfm|php|asp|aspx|jsp)/i,    // cart.html or carts.html
                /checkouts?\.(html?|cfm|php|asp|aspx|jsp)/i,// checkout.html or checkouts.html
                /baskets?\.(html?|cfm|php|asp|aspx|jsp)/i   // basket.html or baskets.html
            ];

            // Check URL patterns (urlLower already declared above)
            for (const pattern of cartUrlPatterns) {
                if (pattern.test(urlLower)) {
                    void 0;
                    return true;
                }
            }
            
            // Page title patterns
            const titlePatterns = [
                /shopping\s*cart/i,
                /\bcart\b.*items?/i,
                /your\s*cart/i,
                /\bbasket\b/i,
                /checkout/i,
                /review.*order/i,
                /order.*summary/i
            ];
            
            if (pageTitle) {
                for (const pattern of titlePatterns) {
                    if (pattern.test(pageTitle)) {
                        void 0;
                        return true;
                    }
                }
            }
            
            // Content-based cart detection DISABLED - causes false positives
            // These patterns (cart total, update cart, proceed to checkout) appear on
            // EVERY page of e-commerce sites in header/mini-cart widgets, not just cart pages.
            // Cart detection must rely on URL and title patterns only.

            return false;
        } catch (error) {
            console.error('Error in cart URL detection:', error);
            return false;
        }
    }

    /**
     * Check if a URL is a product page based on URL patterns
     */
    isProductPage(url) {
        const urlLower = url.toLowerCase();

        // Exclude cart URLs first - they should never be product pages
        // Includes /ViewCart pattern (Uline uses /Product/ViewCart as cart URL)
        if (/\/(cart|basket|checkout|bag)s?\b/i.test(urlLower) || /\/viewcart\b/i.test(urlLower)) {
            return false;
        }

        const productPatterns = [
            /\/dp\//,           // Amazon
            /\/product[\/-]/,   // General: /product/ or /product- (cdgsa uses product-p-123)
            /\/item\//,         // Etsy, eBay
            /\/p\//,            // Target
            /\/products\//,     // Shopify
            /\/buy\//,          // Walmart
            /\/shop\//,         // General shop (but excluded carts above)
            /\/store\//,        // General store
            /\/\w{2}\/\d{4,}-[a-z0-9-]+\.html$/i,  // Intl product: /en/612682-product-name.html (Alpinstore)
            /-g-\d{10,}\.html$/i,  // Temu product: product-name-g-601100056165743.html
            /[?&]productId=/i,  // Query param: productId= (common product identifier)
        ];

        return productPatterns.some(pattern => pattern.test(urlLower));
    }

    /**
     * Check if a URL is an order page (past purchases/order history)
     * Supports multi-language detection across 7 languages
     */
    isOrderPage(url, pageTitle = '') {
        try {
            const urlLower = url.toLowerCase();

            // FIRST: Exclude known product page URL patterns
            // These should NEVER be classified as order pages regardless of title
            const PRODUCT_PAGE_PATTERNS = [
                '/dp/',           // Amazon product detail
                '/gp/product/',   // Amazon alternate product URL
                '/product/',      // Generic product pages
                '/products/',     // Product listings
                '/item/',         // eBay, other sites
                '/p/',            // Target, other sites
                '/ip/',           // Walmart product
                '/pd/',           // Various retailers
                '/shop/',         // Shop pages
                '/buy/'           // Buy pages
            ];

            for (const pattern of PRODUCT_PAGE_PATTERNS) {
                if (urlLower.includes(pattern)) {
                    void 0;
                    return { is_order_page: false, reason: 'product_url_pattern' };
                }
            }

            // SECOND: Exclude product pages with product ID patterns in URL
            // Temu: /product-name-g-XXXXX.html
            // Generic: .html with long numeric ID
            const PRODUCT_ID_PATTERNS = [
                /-g-\d{10,}\.html/i,           // Temu product pages (-g-601100056165743.html)
                /\/[a-z0-9-]+-\d{8,}\.html/i,  // Generic product with ID in filename
            ];

            for (const pattern of PRODUCT_ID_PATTERNS) {
                if (pattern.test(urlLower)) {
                    void 0;
                    return { is_order_page: false, reason: 'product_id_pattern' };
                }
            }

            // THIRD: Only check the URL PATH for order terms, not query parameters
            // Query params often contain terms like "_oak_gallery_order" which are not order pages
            let urlPath = urlLower;
            try {
                const urlObj = new URL(url);
                urlPath = urlObj.pathname.toLowerCase();
            } catch (e) {
                // If URL parsing fails, use the full URL but be more conservative
                urlPath = urlLower.split('?')[0];
            }

            // Multi-language order patterns
            const ORDER_PATTERNS = {
                purchase_order: {
                    en: ['order', 'orders', 'purchase', 'receipt', 'your-account', 'history'],
                    es: ['pedido', 'pedidos', 'compra', 'recibo', 'tu-cuenta', 'historial'],
                    fr: ['commande', 'commandes', 'achat', 'reçu', 'compte', 'historique'],
                    de: ['bestellung', 'bestellungen', 'kauf', 'quittung', 'konto', 'verlauf'],
                    zh: ['订单', '购买', '收据', '账户', '历史'],
                    hi: ['आदेश', 'खरीद', 'रसीद', 'खाता', 'इतिहास'],
                    ar: ['طلب', 'طلبات', 'شراء', 'إيصال', 'حساب', 'تاريخ']
                },
                invoice_order: {
                    en: ['invoice', 'invoices', 'bill', 'billing', 'statement'],
                    es: ['factura', 'facturas', 'cuenta', 'estado'],
                    fr: ['facture', 'factures', 'relevé'],
                    de: ['rechnung', 'rechnungen', 'abrechnung'],
                    zh: ['发票', '账单', '对账单'],
                    hi: ['चालान', 'बिल', 'विवरण'],
                    ar: ['فاتورة', 'فواتير', 'بيان']
                },
                quotation_order: {
                    en: ['quote', 'quotes', 'quotation', 'estimate', 'proposal'],
                    es: ['cotización', 'presupuesto', 'propuesta'],
                    fr: ['devis', 'proposition', 'estimation'],
                    de: ['angebot', 'angebote', 'kostenvoranschlag'],
                    zh: ['报价', '估价', '建议书'],
                    hi: ['उद्धरण', 'अनुमान', 'प्रस्ताव'],
                    ar: ['عرض', 'تقدير', 'اقتراح']
                }
            };

            // Check each document type - only match against URL PATH (not query params)
            for (const [docType, languages] of Object.entries(ORDER_PATTERNS)) {
                for (const [lang, terms] of Object.entries(languages)) {
                    for (const term of terms) {
                        if (urlPath.includes(term)) {
                            void 0;
                            return {
                                is_order_page: true,
                                document_type: docType,
                                language: lang,
                                matched_term: term
                            };
                        }
                    }
                }
            }

            // Check page title if provided
            if (pageTitle) {
                const titleLower = pageTitle.toLowerCase();
                for (const [docType, languages] of Object.entries(ORDER_PATTERNS)) {
                    for (const [lang, terms] of Object.entries(languages)) {
                        for (const term of terms) {
                            if (titleLower.includes(term)) {
                                void 0;
                                return {
                                    is_order_page: true,
                                    document_type: docType,
                                    language: lang,
                                    matched_term: term
                                };
                            }
                        }
                    }
                }
            }

            return { is_order_page: false };
        } catch (error) {
            console.error('Error in order page detection:', error);
            return { is_order_page: false };
        }
    }

    /**
     * Extract order number from URL or page title
     * Supports multiple order number formats across different retailers
     */
    extractOrderNumber(url, pageTitle = '') {
        try {
            const textToSearch = `${url} ${pageTitle}`.toLowerCase();

            // Common order number patterns
            const ORDER_NUMBER_PATTERNS = [
                // Standard order patterns
                /order[#\-_\s]*([a-z0-9\-]{6,})/i,
                /order[#\-_\s]*id[:\-_\s]*([a-z0-9\-]{6,})/i,
                /orderid[:\-_\s]*([a-z0-9\-]{6,})/i,

                // Invoice patterns
                /invoice[#\-_\s]*([a-z0-9\-]{6,})/i,
                /invoice[#\-_\s]*number[:\-_\s]*([a-z0-9\-]{6,})/i,

                // Purchase order patterns
                /po[#\-_\s]*([a-z0-9\-]{6,})/i,
                /purchase[#\-_\s]*order[:\-_\s]*([a-z0-9\-]{6,})/i,

                // Quote/quotation patterns
                /quote[#\-_\s]*([a-z0-9\-]{6,})/i,
                /quotation[#\-_\s]*([a-z0-9\-]{6,})/i,

                // Generic reference patterns
                /ref[#\-_\s]*([a-z0-9\-]{6,})/i,
                /reference[:\-_\s]*([a-z0-9\-]{6,})/i,

                // URL parameter patterns
                /[?&]order[id]*=([a-z0-9\-]{6,})/i,
                /[?&]invoice=([a-z0-9\-]{6,})/i,
                /[?&]orderdetails=([a-z0-9\-]{6,})/i,

                // Amazon-style patterns
                /orderid=([a-z0-9\-]{6,})/i,
                /orderId=([A-Z0-9\-]{6,})/,

                // Numeric-only patterns (last resort)
                /\b(\d{8,})\b/  // At least 8 consecutive digits
            ];

            for (const pattern of ORDER_NUMBER_PATTERNS) {
                const match = textToSearch.match(pattern);
                if (match && match[1]) {
                    const orderNumber = match[1].trim();
                    // Validate it's not too long (avoid matching entire URLs)
                    if (orderNumber.length <= 50) {
                        void 0;
                        return orderNumber;
                    }
                }
            }

            void 0;
            return null;
        } catch (error) {
            console.error('Error extracting order number:', error);
            return null;
        }
    }

    /**
     * Check if cart URL has existing mappings
     */
    async checkExistingCartMapping(cartUrl) {
        try {
            const response = await fetch(`${this.getApiBaseUrl()}/validate_url`, {
                method: 'POST',
                headers: this.getAuthHeaders(),
                body: JSON.stringify({ url: cartUrl })
            });
            
            if (response.ok) {
                const data = await response.json();
                if (data.existing_items && data.existing_items.length > 0) {
                    // Check if any existing items are from cart sessions (multiple items)
                    const cartSessions = data.existing_items.filter(item => 
                        item.session_id && item.session_id.startsWith('ext_')
                    );
                    
                    if (cartSessions.length > 0) {
                        return {
                            hasExisting: true,
                            existingItems: cartSessions,
                            cartResult: {
                                success: true,
                                items: cartSessions,
                                session_id: cartSessions[0].session_id,
                                total_footprint: cartSessions.reduce((sum, item) => 
                                    sum + (item.product_carbon_footprint || 0), 0),
                                item_count: cartSessions.length,
                                platform: 'existing_mapping',
                                processing_stats: {
                                    total_items: cartSessions.length,
                                    cache_hit_rate: 1.0,
                                    existing_mapping_used: true
                                }
                            }
                        };
                    }
                }
            }
            
            return null;
        } catch (error) {
            console.error('CarbonGuru: Error checking existing cart mapping:', error);
            return null;
        }
    }
    
    /**
     * Show dialog for cart remapping choice
     */
    async showCartRemapDialog(existingItems, cartUrl) {
        try {
            const tabs = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).query({active: true, currentWindow: true});
            if (tabs[0]) {
                return new Promise((resolve) => {
                    (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tabs[0].id, {
                        action: 'showExistingMappingsDialog',
                        existingItems: existingItems,
                        url: cartUrl,
                        isCart: true
                    }, (response) => {
                        resolve(response ? response.choice : 'use_existing');
                    });
                });
            }
        } catch (error) {
            console.error('CarbonGuru: Error showing cart remap dialog:', error);
        }
        
        return 'use_existing'; // Default to existing mapping
    }
    
    /**
     * Fallback method: Process cart URL as single item if cart processing fails
     */
    async fallbackToSingleItemProcessing(cartData) {
        void 0;
        
        // Process the cart URL as a single product
        const productData = {
            url: cartData.url,
            full_page_content: cartData.htmlContent,
            page_title: cartData.title,
            extracted_at: cartData.timestamp,
            extraction_method: 'cart_fallback'
        };
        
        return new Promise((resolve, reject) => {
            this.processProductWithExtraction(productData, (result) => {
                if (result.success) {
                    // Wrap single item result in cart format
                    resolve({
                        success: true,
                        items: [result],
                        total_footprint: result.product_carbon_footprint || 0,
                        processing_stats: {
                            total_items: 1,
                            new_items_processed: 1,
                            cache_hit_rate: 0,
                            fallback_used: true
                        },
                        processing_time_ms: result.processing_time_ms || 0,
                        platform: cartData.platform,
                        fallback_method: 'single_item'
                    });
                } else {
                    reject(new Error(result.error || 'Fallback processing failed'));
                }
            });
        });
    }
}

// Add this listener at the top level of the script
const runtime = typeof browser !== 'undefined' ? browser.runtime : chrome.runtime;
runtime.onInstalled.addListener(async (details) => {
    if (details.reason === 'install' || details.reason === 'update') {
        // Show data consent page on first install, or on update if consent not yet given
        try {
            const storage = typeof browser !== 'undefined' ? browser.storage : chrome.storage;
            const result = await storage.local.get('dataConsentGiven');
            if (result.dataConsentGiven === undefined || result.dataConsentGiven === null) {
                const consentUrl = runtime.getURL('consent.html');
                const tabs = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
                await tabs.create({ url: consentUrl });
            }
        } catch (e) {
            void 0;
        }
    }
    if (details.reason === 'update') {
        void 0;
        
        // Ensure content scripts are injected post-update to avoid stale states
        try {
            const tabsAPI = typeof browser !== 'undefined' ? browser.tabs : chrome.tabs;
            const tabsList = await tabsAPI.query({ url: ['https://*/*'] });
            for (const tab of tabsList) {
                try { await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).sendMessage(tab.id, { action: 'ping' }); }
                catch (_) { if (tabsAPI.executeScript) await tabsAPI.executeScript(tab.id, { file: 'content.js' }); }
            }
        } catch (e) {
            void 0;
        }
        
        try {
            const tabs = await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).query({
                url: ['https://*/*']
            });
            
            for (const tab of tabs) {
                if (tab.id) {
                    try {
                        await (typeof browser !== 'undefined' ? browser.tabs : chrome.tabs).reload(tab.id);
                    } catch (e) {
                        console.error(`CarbonGuru: Failed to reload tab ${tab.id}:`, e);
                    }
                }
            }
        } catch (error) {
            console.error('CarbonGuru: Error reloading tabs after update:', error);
        }
    }
});

/**
 * WooCommerceBasketManager - Pure view of WooCommerce cart (no local storage)
 * All operations directly modify WooCommerce cart via Store API
 */
class WooCommerceBasketManager {
    constructor() {
        this.pollIntervalId = null;
        this.pollInterval = 30000; // 30 seconds
        this.costCache = new Map();
        this.COST_CACHE_TTL = 5 * 60 * 1000; // 5 minutes
        void 0;
    }

    /**
     * Get backend URL from CarbonGuruBackground instance
     */
    getBackendUrl() {
        return typeof carbonGuruBackground !== 'undefined'
            ? carbonGuruBackground.backendUrl
            : 'https://carbonguru.io';
    }

    /**
     * Get WordPress cookies for session
     */
    async getWordPressCookies() {
        const backendUrl = this.getBackendUrl();
        const wordpressDomain = new URL(backendUrl).hostname;
        const allCookies = await browser.cookies.getAll({ domain: wordpressDomain });

        const wpCookies = {};
        for (const cookie of allCookies) {
            if (cookie.name.includes('wordpress') ||
                cookie.name.includes('woocommerce') ||
                cookie.name === 'wp_woocommerce_session') {
                wpCookies[cookie.name] = cookie.value;
            }
        }

        void 0;
        if (Object.keys(wpCookies).length > 0) {
            void 0;
        }

        return wpCookies;
    }

    /**
     * Add item to WooCommerce cart via backend sync endpoint
     * @param {Object} item - Item to add (must have id/carbonguru_id)
     * @returns {Object} - { success: boolean, message: string }
     */
    async addItem(item) {
        try {
            if (!item.id && !item.carbonguru_id) {
                console.error('CarbonGuru: Cannot add item - missing id:', item);
                return { success: false, message: 'Missing item ID' };
            }

            const backendUrl = this.getBackendUrl();
            const itemId = item.id || item.carbonguru_id;
            void 0;

            // Use backend sync endpoint which handles product creation and cart addition
            const response = await carbonGuruBackground.authenticatedFetch(
                `${backendUrl}/api/extension/offset-basket/sync`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        items: [{
                            id: itemId,
                            product_name: item.product_name || item.name,
                            product_carbon_footprint: item.product_carbon_footprint || 0,
                            social_cost_usd: item.social_cost_usd || 0,
                            product_url: item.product_url || '',
                            product_image: item.product_image || '',
                            item_type: item.item_type || 'product',
                            // Pass preference type and product IDs for offset vs removal selection
                            preference_type: item.preference_type || 'offset',
                            preferred_product_id: item.preferred_product_id || null,
                            preferred_climate_product_id: item.preferred_climate_product_id || null,
                            // Pass fee-included cost from quote service for accurate cart totals
                            fee_included_cost: item.fee_included_cost || null
                        }],
                        wordpress_cookies: await this.getWordPressCookies()
                    })
                }
            );

            if (!response.ok) {
                const error = await response.json();
                throw new Error(error.error || 'Failed to add item');
            }

            const data = await response.json();

            // Trust Flask's success field - if Flask says success, it's success
            if (data.success) {
                return { success: true, message: 'Item added to cart' };
            } else {
                const errorMsg = data.errors?.[0]?.error || data.errors?.[0] || data.message || 'Failed to add item';
                return { success: false, message: errorMsg };
            }
        } catch (error) {
            console.error('CarbonGuru: Error adding item to WooCommerce cart:', error);
            return { success: false, message: error.message };
        }
    }

    /**
     * Remove item from WooCommerce cart
     * @param {string} cartItemKey - WooCommerce cart_item_key
     * @returns {Object} - { success: boolean, message: string }
     */
    async removeItem(cartItemKey) {
        try {
            if (!cartItemKey) {
                console.error('CarbonGuru: Cannot remove item - missing cart_item_key');
                return { success: false, message: 'Missing cart item key' };
            }

            const backendUrl = this.getBackendUrl();
            void 0;

            // Call backend endpoint with JWT authentication and WordPress cookies
            // This matches the pattern used in fetchWooCommerceCart() for proper WordPress session management
            const response = await carbonGuruBackground.authenticatedFetch(
                `${backendUrl}/api/extension/offset-basket/remove-item`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        cart_item_key: cartItemKey,
                        wordpress_cookies: await this.getWordPressCookies()
                    })
                }
            );

            if (!response.ok) {
                throw new Error(`Failed to remove item: ${response.status}`);
            }

            const data = await response.json();

            if (data.success) {
                void 0;
                return { success: true, message: data.message || 'Item removed from cart' };
            } else {
                console.error('CarbonGuru: Failed to remove item:', data.error);
                return { success: false, message: data.error || 'Failed to remove item' };
            }
        } catch (error) {
            console.error('CarbonGuru: Error removing item from WooCommerce cart:', error);
            return { success: false, message: error.message };
        }
    }

    /**
     * Get basket from WooCommerce (no local storage)
     * @returns {Promise<Object>} - Cart data from WooCommerce
     */
    async getBasket() {
        return await this.fetchWooCommerceCart();
    }

    /**
     * Get basket summary for display (fetches from WooCommerce)
     * @returns {Promise<Object>} - Basket summary with items, count, totals
     */
    async getBasketSummary() {
        const wcCart = await this.fetchWooCommerceCart();

        if (!wcCart.success) {
            return {
                items: [],
                count: 0,
                totalFootprint: 0,
                totalFootprintFormatted: '0 kg CO2e',
                totalCarbonCost: 0,
                totalCarbonCostFormatted: null,
                cart_url: `${this.getBackendUrl()}/cart/`,
                checkout_url: `${this.getBackendUrl()}/checkout/`,
                error: wcCart.error
            };
        }

        // Calculate totals from individual items (fee-included costs)
        // This ensures accurate totals regardless of backend calculation
        const items = wcCart.cart_items || [];
        let calculatedOffsetCost = 0;
        let calculatedRemovalCost = 0;

        for (const item of items) {
            const quantity = item.quantity || 1;
            // Get the item's cost - prefer fee_included_cost, then offset_cost_usd, then display_cost_usd
            const itemCost = parseFloat(item.fee_included_cost || item.offset_cost_usd || item.display_cost_usd || item.orderable_cost_usd || 0);
            const totalItemCost = itemCost * quantity;

            if (item.preference_type === 'removal') {
                calculatedRemovalCost += totalItemCost;
            } else {
                calculatedOffsetCost += totalItemCost;
            }
        }

        return {
            items: items,
            count: wcCart.cart_count || 0,
            totalFootprint: wcCart.total_footprint || 0,
            totalFootprintFormatted: this.formatFootprint(wcCart.total_footprint || 0),
            totalCarbonCost: wcCart.total_carbon_cost || 0,
            totalCarbonCostFormatted: wcCart.total_carbon_cost > 0
                ? `$${wcCart.total_carbon_cost.toFixed(2)}`
                : null,
            totalOffsetCost: calculatedOffsetCost,
            totalOffsetCostFormatted: calculatedOffsetCost > 0
                ? `$${calculatedOffsetCost.toFixed(2)}`
                : null,
            totalRemovalCost: calculatedRemovalCost,
            totalRemovalCostFormatted: calculatedRemovalCost > 0
                ? `$${calculatedRemovalCost.toFixed(2)}`
                : null,
            cart_url: wcCart.cart_url,
            checkout_url: wcCart.checkout_url || `${this.getBackendUrl()}/checkout/`,
            rawCart: wcCart
        };
    }

    /**
     * Format carbon footprint for display
     * @param {number} footprint - Footprint in kg CO2e
     * @returns {string} - Formatted footprint string
     */
    formatFootprint(footprint) {
        if (footprint === 0) return '0 kg CO2e';
        if (footprint < 0.01) return '<0.01 kg CO2e';
        if (footprint < 1) return `${footprint.toFixed(2)} kg CO2e`;
        if (footprint < 1000) return `${footprint.toFixed(1)} kg CO2e`;
        return `${(footprint / 1000).toFixed(1)} t CO2e`;
    }

    /**
     * Start background polling to keep cart updated
     */
    startPolling() {
        if (this.pollIntervalId) {
            void 0;
            return;
        }

        void 0;
        this.pollIntervalId = setInterval(async () => {
            const cart = await this.fetchWooCommerceCart();
            void 0;
        }, this.pollInterval);
    }

    /**
     * Stop background polling
     */
    stopPolling() {
        if (this.pollIntervalId) {
            clearInterval(this.pollIntervalId);
            this.pollIntervalId = null;
            void 0;
        }
    }

    /**
     * Open WooCommerce cart page
     */
    async openCart() {
        const cart = await this.fetchWooCommerceCart();
        if (cart.success && cart.cart_url) {
            void 0;
            browser.tabs.create({ url: cart.cart_url });
        } else {
            console.error('CarbonGuru: Failed to get cart URL');
        }
    }

    /**
     * Fetch current WooCommerce cart contents directly from WordPress REST API
     * @returns {Promise<Object>} - Cart data with items, counts, totals
     */
    async fetchWooCommerceCart() {
        try {
            void 0;

            const backendUrl = this.getBackendUrl();

            // Use backend endpoint which handles cookie forwarding to WordPress
            // This matches the pattern used in addItem() which works successfully
            const response = await carbonGuruBackground.authenticatedFetch(
                `${backendUrl}/api/extension/offset-basket/get-cart`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    cache: 'no-store',
                    body: JSON.stringify({
                        wordpress_cookies: await this.getWordPressCookies()
                    })
                }
            );

            if (!response.ok) {
                const error = await response.json();
                throw new Error(error.error || 'Failed to fetch cart');
            }

            const data = await response.json();
            data._fetched_from = backendUrl;
            data._fetched_at = new Date().toISOString();

            const firstItem = Array.isArray(data.cart_items) && data.cart_items.length > 0
                ? data.cart_items[0]
                : null;

            void 0;

            return data;
        } catch (error) {
            console.error('CarbonGuru: Error fetching WooCommerce cart:', error);
            return { success: false, error: error.message };
        }
    }

    // ==========================================
    // Offset/Removal Cost Methods
    // ==========================================

    /**
     * Fetch user preferences via Flask backend (cached for 10 min)
     * Returns offset and removal product preferences.
     *
     * Routes through Flask with explicit cookie passing for cross-browser
     * reliability (Chrome MV3 service workers, Firefox MV2, Safari MV3).
     */
    async getUserPreferences() {
        // Check cache
        if (this.userPreferencesCache &&
            this.userPreferencesCacheTime &&
            Date.now() - this.userPreferencesCacheTime < this.USER_PREFS_CACHE_TTL) {
            void 0;
            return this.userPreferencesCache;
        }

        try {
            void 0;
            const backendUrl = this.getBackendUrl();
            const response = await carbonGuruBackground.authenticatedFetch(
                `${backendUrl}/api/extension/user/preferences`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    cache: 'no-store',
                    body: JSON.stringify({
                        wordpress_cookies: await this.getWordPressCookies()
                    })
                }
            );

            if (!response.ok) {
                void 0;
                return null;
            }

            const prefs = await response.json();
            void 0;

            this.userPreferencesCache = prefs;
            this.userPreferencesCacheTime = Date.now();
            return prefs;
        } catch (error) {
            void 0;
            return null;
        }
    }

    /**
     * Fetch user's carbon impact summary via Flask backend (cached for 5 min)
     * Returns footprint % addressed, offset/removal totals, and recent orders.
     * Used by extension popup to display the impact widget.
     *
     * Routes through Flask with explicit cookie passing for cross-browser
     * reliability (Chrome MV3 service workers, Firefox MV2, Safari MV3).
     */
    async getUserImpactData() {
        // Check cache (5 minute TTL)
        if (this.impactDataCache &&
            this.impactDataCacheTime &&
            Date.now() - this.impactDataCacheTime < 300000) {
            void 0;
            return this.impactDataCache;
        }

        try {
            void 0;
            const backendUrl = this.getBackendUrl();
            const response = await carbonGuruBackground.authenticatedFetch(
                `${backendUrl}/api/extension/user/impact`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    cache: 'no-store',
                    body: JSON.stringify({
                        wordpress_cookies: await this.getWordPressCookies()
                    })
                }
            );

            if (!response.ok) {
                void 0;
                return null;
            }

            const data = await response.json();
            void 0;

            this.impactDataCache = data;
            this.impactDataCacheTime = Date.now();
            return data;
        } catch (error) {
            void 0;
            return null;
        }
    }

    /**
     * Get offset and removal costs for a carbon amount
     * @param {number} carbon_kg - Carbon footprint in kg
     * @returns {Promise<Object>} - { success, offset_cost_usd, removal_cost_usd, ... }
     */
    async getOffsetCosts(carbon_kg) {
        const cacheKey = `cost_${carbon_kg}`;
        const cached = this.getCachedCost(cacheKey);
        if (cached) {
            void 0;
            return cached;
        }

        try {
            void 0;

            // 1. Get user preferences (contains product IDs)
            const prefs = await this.getUserPreferences();

            // 2. Get offset quote
            const offsetProductId = prefs?.offset?.climate_product_id || null;
            const offsetQuote = await this.getQuoteFromService(carbon_kg, 'offset', offsetProductId);

            // 3. Get removal quote
            const removalProductId = prefs?.removal?.climate_product_id || null;
            const removalQuote = await this.getQuoteFromService(carbon_kg, 'removal', removalProductId);

            // If BOTH quotes failed (null), treat as a failure — don't cache or report success
            // This prevents caching {success: true, offset_cost_usd: 0} when the API was unreachable
            if (!offsetQuote && !removalQuote) {
                void 0;
                return { success: false, error: 'Both quote requests failed' };
            }

            const result = {
                success: true,
                offset_cost_usd: offsetQuote?.cost_usd || 0,
                offset_product_id: offsetProductId,
                offset_product_name: prefs?.offset?.product_name || 'Carbon Offset',
                offset_woo_product_id: prefs?.offset?.woo_product_id || null,
                removal_cost_usd: removalQuote?.cost_usd || 0,
                removal_product_id: removalProductId,
                removal_product_name: prefs?.removal?.product_name || 'Carbon Removal',
                removal_woo_product_id: prefs?.removal?.woo_product_id || null,
                has_preferences: !!(prefs?.offset || prefs?.removal),
                carbon_kg: carbon_kg
            };

            void 0;

            // Only cache if we got at least one valid cost
            if (result.offset_cost_usd > 0 || result.removal_cost_usd > 0) {
                this.setCachedCost(cacheKey, result);
            } else {
                void 0;
            }
            return result;
        } catch (error) {
            console.error('CarbonGuru: Error fetching costs:', error);
            return { success: false, error: error.message };
        }
    }

    /**
     * Call offset cost service for a single quote
     * @param {number} carbon_kg - Carbon footprint in kg
     * @param {string} preference - 'offset' or 'removal'
     * @param {string|null} product_id - Specific product ID (from user preferences)
     * @returns {Promise<Object|null>} - Quote response or null on error
     */
    async getQuoteFromService(carbon_kg, preference, product_id) {
        const payload = {
            total_carbon_kg: carbon_kg,  // API expects total_carbon_kg
            preference: preference,
            include_fees: true
        };

        if (product_id) {
            payload.product_ids = [product_id];
        }

        try {
            void 0;

            // Call offset cost service through backend proxy (use global carbonGuruBackground instance)
            const response = await carbonGuruBackground.authenticatedFetch(
                `${carbonGuruBackground.backendUrl}/api/extension/offset-basket/quote`,
                {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify(payload)
                }
            );

            if (!response.ok) {
                void 0;
                return null;
            }

            const data = await response.json();

            // API returns the user's preferred product in 'selected' array
            // Note: Stripe Climate products are always type "removal" even if user selected for offset preference
            // Use the first selected item since API already applies user preferences
            let quote = null;
            if (data.selected && Array.isArray(data.selected) && data.selected.length > 0) {
                quote = data.selected[0];
            }

            // Use total_cost_usd (fee-included when include_fees=true) instead of item's cost_usd
            // When fees_included=true: total_cost_usd = grand_total_usd = subtotal + fees
            // The item's cost_usd is just the pre-fee display cost
            // Prefer precise values for sub-cent amounts (< $0.01)
            const feeIncludedCost = data.total_cost_usd_precise || data.total_cost_usd || quote?.cost_usd_precise || quote?.cost_usd || 0;

            void 0;

            // Return the quote with fee-included cost_usd at top level
            return quote ? { ...quote, cost_usd: feeIncludedCost, success: true } : null;
        } catch (error) {
            console.error('CarbonGuru: Error getting quote from service:', error);
            return null;
        }
    }

    /**
     * Add item to basket with specific offset/removal type
     * @param {Object} item - Item data from analysis
     * @param {string} preferenceType - 'offset' or 'removal'
     */
    async addToBasketWithType(item, preferenceType) {
        try {
            void 0;

            // Get user preferences to find the right product
            const prefs = await this.getUserPreferences();

            // Determine which product to add based on preference type
            let productConfig;
            if (preferenceType === 'removal') {
                productConfig = prefs?.removal || null;
            } else {
                productConfig = prefs?.offset || null;
            }

            // Prepare basket item with preference type
            const basketItem = {
                ...item,
                preference_type: preferenceType,
                preferred_product_id: productConfig?.woo_product_id || null,
                preferred_climate_product_id: productConfig?.climate_product_id || null
            };

            // Use existing addItem method (this is WooCommerceBasketManager, not CarbonGuruBackground)
            const result = await this.addItem(basketItem);
            // Return the actual success status from addItem - don't override with success: true
            // (previous bug: { success: true, ...result } would have success overwritten by spread)
            return result;
        } catch (error) {
            console.error('CarbonGuru: Error adding to basket with type:', error);
            return { success: false, error: error.message };
        }
    }

    /**
     * Get cached cost data
     */
    getCachedCost(key) {
        const entry = this.costCache.get(key);
        if (entry && Date.now() - entry.time < this.COST_CACHE_TTL) {
            return entry.data;
        }
        return null;
    }

    /**
     * Set cached cost data
     */
    setCachedCost(key, data) {
        this.costCache.set(key, { data, time: Date.now() });
    }

}

// Main instantiation of the background script
const carbonGuruBackground = new CarbonGuruBackground(); 
